// =============================================
// SOLVO — Settings Page Controller
// =============================================

const PROVIDER_INFO = {
  free:       { label: 'Free AI',       needsKey: false, keyField: null, helpUrl: null, steps: [] },
  gemini:     { label: 'Google Gemini', needsKey: true,  keyField: 'apiKey',
    helpUrl: 'https://aistudio.google.com/app/apikey',
    steps: ['Go to Google AI Studio', 'Sign in with Google', 'Click "Create API Key"', 'Copy and paste here'] },
  groq:       { label: 'Groq',          needsKey: true,  keyField: 'groqKey',
    helpUrl: 'https://console.groq.com/keys',
    steps: ['Visit console.groq.com', 'Create a free account', 'Go to API Keys', 'Create & copy your key'] },
  openrouter: { label: 'OpenRouter',    needsKey: true,  keyField: 'openrouterKey',
    helpUrl: 'https://openrouter.ai/keys',
    steps: ['Go to openrouter.ai', 'Sign up for free', 'Navigate to Keys', 'Create a key and copy it'] },
  together:   { label: 'Together AI',   needsKey: true,  keyField: 'togetherKey',
    helpUrl: 'https://api.together.xyz/settings/api-keys',
    steps: ['Visit together.ai', 'Create a free account', 'Settings → API Keys', 'Generate & copy'] },
};

const MODELS = {
  gemini: [
    { value: 'gemini-2.0-flash', label: 'Gemini 2.0 Flash' },
    { value: 'gemini-2.0-flash-lite', label: 'Gemini 2.0 Flash Lite' },
    { value: 'gemini-1.5-flash', label: 'Gemini 1.5 Flash' },
    { value: 'gemini-1.5-flash-8b', label: 'Gemini 1.5 Flash 8B' },
    { value: 'gemini-1.5-pro', label: 'Gemini 1.5 Pro' },
  ],
  groq: [
    { value: 'llama-3.3-70b-versatile', label: 'Llama 3.3 70B Versatile' },
    { value: 'llama-3.1-8b-instant', label: 'Llama 3.1 8B Instant' },
    { value: 'mixtral-8x7b-32768', label: 'Mixtral 8x7B' },
    { value: 'gemma2-9b-it', label: 'Gemma 2 9B' },
  ],
  openrouter: [
    { value: 'meta-llama/llama-3.3-70b-instruct:free', label: 'Llama 3.3 70B (free)' },
    { value: 'google/gemini-2.0-flash-exp:free', label: 'Gemini 2.0 Flash (free)' },
    { value: 'deepseek/deepseek-r1:free', label: 'DeepSeek R1 (free)' },
    { value: 'qwen/qwen-2.5-72b-instruct:free', label: 'Qwen 2.5 72B (free)' },
  ],
  together: [
    { value: 'meta-llama/Llama-3.3-70B-Instruct-Turbo', label: 'Llama 3.3 70B Turbo' },
    { value: 'Qwen/Qwen2.5-72B-Instruct-Turbo', label: 'Qwen 2.5 72B Turbo' },
    { value: 'deepseek-ai/DeepSeek-R1-Distill-Llama-70B', label: 'DeepSeek R1 70B' },
  ],
};

const $ = (sel) => document.querySelector(sel);
let savedSnapshot = '';
let toastTimer = null;

document.addEventListener('DOMContentLoaded', init);

async function init() {
  // Set version from manifest
  try {
    const ver = chrome.runtime.getManifest().version;
    const versionEl = document.getElementById('ext-version');
    if (versionEl) versionEl.textContent = 'v' + ver;
  } catch(e) {}

  await loadSettings();
  bindEvents();
  computeStorageSize();
  savedSnapshot = getSnapshot();
}

// ---- Load ----
async function loadSettings() {
  const data = await chrome.storage.local.get([
    'provider', 'apiKey', 'groqKey', 'openrouterKey', 'togetherKey',
    'model', 'batchSize', 'batchDelay', 'profile',
    'enableRightClick', 'enableCopyPaste', 'enableTabSwitch', 'enableFullscreenBypass', 'enableBrowserSpoof',
  ]);

  const provider = data.provider || 'free';
  $('#settings-provider').value = provider;

  // API key — load the correct key for the selected provider (no cross-provider fallback)
  const info = PROVIDER_INFO[provider];
  if (info.needsKey && info.keyField) {
    $('#settings-api-key').value = data[info.keyField] || '';
  }

  // Model
  if (MODELS[provider]) {
    populateModels(provider, data.model);
  }

  // Rate limiting
  $('#settings-batch-size').value = data.batchSize || 5;
  $('#settings-batch-delay').value = data.batchDelay || 1200;

  // Profile
  const profile = data.profile || {};
  $('#s-profile-name').value = profile.name || '';
  $('#s-profile-email').value = profile.email || '';
  $('#s-profile-phone').value = profile.phone || '';
  $('#s-profile-org').value = profile.org || '';
  $('#s-profile-id').value = profile.id || '';
  $('#s-profile-address').value = profile.address || '';

  // Bypass features
  const rcEl = $('#s-enable-right-click');
  const cpEl = $('#s-enable-copy-paste');
  const tsEl = $('#s-enable-tab-switch');
  const fsEl = $('#s-enable-fullscreen-bypass');
  const bsEl = $('#s-enable-browser-spoof');
  if (rcEl) rcEl.checked = !!data.enableRightClick;
  if (cpEl) cpEl.checked = !!data.enableCopyPaste;
  if (tsEl) tsEl.checked = !!data.enableTabSwitch;
  if (fsEl) fsEl.checked = !!data.enableFullscreenBypass;
  if (bsEl) bsEl.checked = !!data.enableBrowserSpoof;

  updateProviderUI(provider);
}

function updateProviderUI(provider) {
  const info = PROVIDER_INFO[provider];
  const keySection = $('#settings-key-section');
  const modelSection = $('#settings-model-section');

  if (info.needsKey) {
    keySection.style.display = '';
    $('#settings-key-label').textContent = `${info.label} API Key`;

    // Steps
    const ol = $('#settings-key-steps');
    ol.innerHTML = '';
    info.steps.forEach(s => {
      const li = document.createElement('li');
      li.textContent = s;
      ol.appendChild(li);
    });

    // Help link
    const link = $('#settings-key-link');
    if (info.helpUrl) {
      link.href = info.helpUrl;
      link.style.display = '';
    } else {
      link.style.display = 'none';
    }
  } else {
    keySection.style.display = 'none';
  }

  if (MODELS[provider]) {
    modelSection.style.display = '';
    populateModels(provider);
  } else {
    modelSection.style.display = 'none';
  }
}

function populateModels(provider, selected) {
  const sel = $('#settings-model');
  sel.innerHTML = '';
  (MODELS[provider] || []).forEach(m => {
    const opt = document.createElement('option');
    opt.value = m.value;
    opt.textContent = m.label;
    sel.appendChild(opt);
  });
  if (selected) sel.value = selected;
}

// Load the correct API key for a given provider from storage
async function loadKeyForProvider(provider) {
  const info = PROVIDER_INFO[provider];
  if (!info.needsKey) {
    $('#settings-api-key').value = '';
    return;
  }
  const data = await chrome.storage.local.get([info.keyField]);
  $('#settings-api-key').value = data[info.keyField] || '';
}

// ---- Events ----
function bindEvents() {
  // Provider change — load correct API key from storage
  $('#settings-provider').addEventListener('change', async (e) => {
    const newProvider = e.target.value;
    await loadKeyForProvider(newProvider);
    updateProviderUI(newProvider);
    checkDirty();
  });

  // Toggle key visibility
  $('#settings-toggle-key').addEventListener('click', () => {
    const inp = $('#settings-api-key');
    inp.type = inp.type === 'password' ? 'text' : 'password';
  });

  // Track changes for save bar
  document.querySelectorAll('input, select').forEach(el => {
    el.addEventListener('input', checkDirty);
    el.addEventListener('change', checkDirty);
  });

  // Save
  $('#btn-save').addEventListener('click', saveSettings);
  $('#btn-discard').addEventListener('click', discardChanges);

  // Clear cache
  $('#btn-clear-cache').addEventListener('click', clearCache);

  // Reset all
  $('#btn-reset-all').addEventListener('click', resetAll);
}

function getSnapshot() {
  const fields = document.querySelectorAll('input, select');
  return Array.from(fields).map(f => {
    if (f.type === 'checkbox') return f.checked ? '1' : '0';
    return f.value;
  }).join('|');
}

function checkDirty() {
  const dirty = getSnapshot() !== savedSnapshot;
  $('#save-bar').classList.toggle('visible', dirty);
}

// ---- Save ----
async function saveSettings() {
  const provider = $('#settings-provider').value;
  const info = PROVIDER_INFO[provider];

  const toSave = {
    provider,
    batchSize: parseInt($('#settings-batch-size').value) || 5,
    batchDelay: parseInt($('#settings-batch-delay').value) || 1200,
    profile: {
      name:    $('#s-profile-name').value.trim(),
      email:   $('#s-profile-email').value.trim(),
      phone:   $('#s-profile-phone').value.trim(),
      org:     $('#s-profile-org').value.trim(),
      id:      $('#s-profile-id').value.trim(),
      address: $('#s-profile-address').value.trim(),
    },
    enableRightClick:      $('#s-enable-right-click')?.checked || false,
    enableCopyPaste:       $('#s-enable-copy-paste')?.checked || false,
    enableTabSwitch:       $('#s-enable-tab-switch')?.checked || false,
    enableFullscreenBypass: $('#s-enable-fullscreen-bypass')?.checked || false,
    enableBrowserSpoof:    $('#s-enable-browser-spoof')?.checked || false,
  };

  // Save API key to correct field
  if (info.needsKey && info.keyField) {
    const key = $('#settings-api-key').value.trim();
    toSave[info.keyField] = key;
    // Also set apiKey for backward compat with gemini
    if (provider === 'gemini') toSave.apiKey = key;
  }

  // Save model
  if (MODELS[provider]) {
    toSave.model = $('#settings-model').value;
  }

  await chrome.storage.local.set(toSave);
  savedSnapshot = getSnapshot();
  $('#save-bar').classList.remove('visible');
  showToast('Settings saved!', 'success');
}

async function discardChanges() {
  await loadSettings();
  savedSnapshot = getSnapshot();
  $('#save-bar').classList.remove('visible');
}

// ---- Clear cache ----
async function clearCache() {
  // Keep essential keys, remove everything else
  const essential = ['setupDone', 'provider', 'apiKey', 'groqKey', 'openrouterKey', 'togetherKey', 'model', 'batchSize', 'batchDelay', 'profile', 'enableRightClick', 'enableCopyPaste', 'enableTabSwitch', 'enableFullscreenBypass', 'enableBrowserSpoof'];
  const all = await chrome.storage.local.get(null);
  const toRemove = Object.keys(all).filter(k => !essential.includes(k));

  if (toRemove.length === 0) {
    showToast('No cache to clear', 'info');
    return;
  }

  await chrome.storage.local.remove(toRemove);

  // Also clear solving state in background
  try {
    await chrome.runtime.sendMessage({ action: 'clearSolvingState' });
  } catch {}

  showToast(`Cleared ${toRemove.length} cached items`, 'success');
  computeStorageSize();
}

// ---- Reset all ----
async function resetAll() {
  if (!confirm('This will erase ALL settings, API keys, and profile data. Continue?')) return;

  await chrome.storage.local.clear();
  try {
    await chrome.runtime.sendMessage({ action: 'clearSolvingState' });
  } catch {}

  showToast('All data cleared. Reloading...', 'success');
  setTimeout(() => window.location.reload(), 1500);
}

// ---- Storage size ----
async function computeStorageSize() {
  try {
    const all = await chrome.storage.local.get(null);
    const json = JSON.stringify(all);
    const bytes = new Blob([json]).size;
    const label = bytes < 1024 ? `${bytes} B` : `${(bytes / 1024).toFixed(1)} KB`;
    $('#storage-size').textContent = label;
  } catch {
    $('#storage-size').textContent = 'Unknown';
  }
}

// ---- Toast ----
function showToast(msg, type = 'info') {
  const toast = $('#settings-toast');
  const icon = $('#settings-toast-icon');
  const msgSpan = $('#settings-toast-msg');

  toast.className = 'toast' + (type ? ' ' + type : '');
  msgSpan.textContent = msg;

  if (type === 'success') {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><polyline points="20 6 9 17 4 12"/></svg>';
  } else if (type === 'error') {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>';
  } else {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>';
  }

  toast.style.display = '';
  if (toastTimer) clearTimeout(toastTimer);
  toastTimer = setTimeout(() => { toast.style.display = 'none'; }, 3500);
}

// ---- How-to-use link ----
const howToBtn = document.getElementById('btn-how-to-use');
if (howToBtn) {
  howToBtn.addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: chrome.runtime.getURL('pages/install.html') });
  });
}
