// =============================================
// SOLVO — Popup Controller v1.2
// Messaging-based: all solve work runs in
// background.js so it survives popup close.
// Settings open in a separate tab.
// =============================================

const PROVIDER_INFO = {
  free:       { label: 'Free AI',      needsKey: false, dot: '#0F7B6C', helpUrl: null, steps: [] },
  gemini:     { label: 'Google Gemini', needsKey: true,  dot: '#4285F4',
    helpUrl: 'https://aistudio.google.com/app/apikey',
    steps: ['Go to Google AI Studio', 'Sign in with your Google account', 'Click "Create API Key"', 'Copy the key and paste it here'] },
  groq:       { label: 'Groq',          needsKey: true,  dot: '#F59E0B',
    helpUrl: 'https://console.groq.com/keys',
    steps: ['Go to console.groq.com', 'Create a free account', 'Navigate to API Keys', 'Click "Create API Key" and copy it'] },
  openrouter: { label: 'OpenRouter',    needsKey: true,  dot: '#8B5CF6',
    helpUrl: 'https://openrouter.ai/keys',
    steps: ['Go to openrouter.ai', 'Sign up for free', 'Go to "Keys" in the sidebar', 'Create a new key and copy it'] },
  together:   { label: 'Together AI',   needsKey: true,  dot: '#EC4899',
    helpUrl: 'https://api.together.xyz/settings/api-keys',
    steps: ['Go to together.ai', 'Create a free account', 'Open Settings → API Keys', 'Generate a key and copy it'] },
};

const MODELS = {
  gemini: [
    { value: 'gemini-2.0-flash', label: 'Gemini 2.0 Flash', default: true },
    { value: 'gemini-2.0-flash-lite', label: 'Gemini 2.0 Flash Lite' },
    { value: 'gemini-1.5-flash', label: 'Gemini 1.5 Flash' },
    { value: 'gemini-1.5-flash-8b', label: 'Gemini 1.5 Flash 8B' },
    { value: 'gemini-1.5-pro', label: 'Gemini 1.5 Pro' },
  ],
  groq: [
    { value: 'llama-3.3-70b-versatile', label: 'Llama 3.3 70B Versatile', default: true },
    { value: 'llama-3.1-8b-instant', label: 'Llama 3.1 8B Instant' },
    { value: 'mixtral-8x7b-32768', label: 'Mixtral 8x7B' },
    { value: 'gemma2-9b-it', label: 'Gemma 2 9B' },
  ],
  openrouter: [
    { value: 'meta-llama/llama-3.3-70b-instruct:free', label: 'Llama 3.3 70B (free)', default: true },
    { value: 'google/gemini-2.0-flash-exp:free', label: 'Gemini 2.0 Flash (free)' },
    { value: 'deepseek/deepseek-r1:free', label: 'DeepSeek R1 (free)' },
    { value: 'qwen/qwen-2.5-72b-instruct:free', label: 'Qwen 2.5 72B (free)' },
  ],
  together: [
    { value: 'meta-llama/Llama-3.3-70B-Instruct-Turbo', label: 'Llama 3.3 70B Turbo', default: true },
    { value: 'Qwen/Qwen2.5-72B-Instruct-Turbo', label: 'Qwen 2.5 72B Turbo' },
    { value: 'deepseek-ai/DeepSeek-R1-Distill-Llama-70B', label: 'DeepSeek R1 70B' },
  ],
};

// ---- DOM Cache ----
const $ = (sel) => document.querySelector(sel);
const $$ = (sel) => document.querySelectorAll(sel);

// ---- State ----
let currentTabId = null;
let pollTimer = null;
let toastTimer = null;

// =============================================
// INIT
// =============================================
document.addEventListener('DOMContentLoaded', init);

async function init() {
  const data = await chrome.storage.local.get(['setupDone', 'provider']);
  if (data.setupDone) {
    showScreen('main');
    await initMain();
  } else {
    showScreen('onboarding');
    initOnboarding();
  }
}

function showScreen(name) {
  $$('.screen').forEach(s => s.style.display = 'none');
  const screen = $(`#${name}-screen`);
  if (screen) screen.style.display = '';
}

// =============================================
// ONBOARDING
// =============================================
function initOnboarding() {
  // Provider card selection
  $$('.provider-option').forEach(opt => {
    opt.addEventListener('click', () => {
      $$('.provider-option').forEach(o => o.classList.remove('selected'));
      opt.classList.add('selected');
      opt.querySelector('input[type="radio"]').checked = true;
      updateKeySection();
    });
  });

  // Key visibility toggle
  const toggleBtn = $('#toggle-key-visibility');
  const keyInput = $('#api-key-input');
  if (toggleBtn && keyInput) {
    toggleBtn.addEventListener('click', () => {
      keyInput.type = keyInput.type === 'password' ? 'text' : 'password';
    });
  }

  // Next step
  $('#btn-next-step').addEventListener('click', async () => {
    const provider = getSelectedProvider();
    const info = PROVIDER_INFO[provider];

    if (info.needsKey) {
      const key = keyInput?.value?.trim();
      if (!key) {
        showToast('Please enter your API key', 'error');
        return;
      }
      const keyField = provider === 'gemini' ? 'apiKey' : `${provider}Key`;
      await chrome.storage.local.set({ provider, [keyField]: key });
    } else {
      await chrome.storage.local.set({ provider });
    }

    // Show step 2
    $('#step-api').classList.remove('active');
    $('#step-profile').classList.add('active');
  });

  // Profile save
  $('#btn-save-profile').addEventListener('click', async () => {
    const profile = gatherProfile();
    await chrome.storage.local.set({ profile, setupDone: true });
    showScreen('main');
    await initMain();
  });

  // Profile skip
  $('#btn-skip-profile').addEventListener('click', async () => {
    await chrome.storage.local.set({ setupDone: true });
    showScreen('main');
    await initMain();
  });

  updateKeySection();
}

function getSelectedProvider() {
  const checked = $('input[name="onb-provider"]:checked');
  return checked ? checked.value : 'free';
}

function updateKeySection() {
  const provider = getSelectedProvider();
  const info = PROVIDER_INFO[provider];
  const section = $('#onb-key-section');

  if (info.needsKey) {
    section.style.display = '';
    $('#onb-key-label').textContent = `${info.label} API Key`;

    // Update guide steps
    const stepsOl = $('#onb-key-steps');
    stepsOl.innerHTML = '';
    info.steps.forEach(step => {
      const li = document.createElement('li');
      li.textContent = step;
      stepsOl.appendChild(li);
    });

    // Update help link
    const helpLink = $('#onb-key-help');
    if (info.helpUrl) {
      helpLink.href = info.helpUrl;
      helpLink.style.display = '';
    } else {
      helpLink.style.display = 'none';
    }
  } else {
    section.style.display = 'none';
  }
}

function gatherProfile() {
  return {
    name:    $('#profile-name')?.value?.trim() || '',
    email:   $('#profile-email')?.value?.trim() || '',
    phone:   $('#profile-phone')?.value?.trim() || '',
    org:     $('#profile-org')?.value?.trim() || '',
    id:      $('#profile-id')?.value?.trim() || '',
    address: $('#profile-address')?.value?.trim() || '',
  };
}

// =============================================
// MAIN SCREEN
// =============================================
async function initMain() {
  // Load AI badge
  const data = await chrome.storage.local.get(['provider']);
  const provider = data.provider || 'free';
  const info = PROVIDER_INFO[provider];
  $('#ai-badge-text').textContent = info.label;
  const dot = $('#ai-badge-dot');
  if (dot) dot.style.background = info.dot;

  // Settings button → opens settings.html in new tab
  $('#btn-settings').addEventListener('click', () => {
    chrome.tabs.create({ url: chrome.runtime.getURL('settings/settings.html') });
  });

  // Init tabs
  initTabs();

  // Check current tab for forms
  await checkCurrentTab();

  // Init link tab
  initLinkTab();

  // Check for ongoing solve
  checkOngoingSolve();

  // Check for updates and show banner if available
  checkAndShowUpdateBanner();
}

// ---- Tabs ----
function initTabs() {
  $$('.tab-btn').forEach(btn => {
    btn.addEventListener('click', () => {
      $$('.tab-btn').forEach(b => b.classList.remove('active'));
      btn.classList.add('active');
      $$('.tab-panel').forEach(p => p.classList.remove('active'));
      const panel = $(`#tab-${btn.dataset.tab}`);
      if (panel) panel.classList.add('active');
    });
  });
}

// ---- Check Current Tab ----
async function checkCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab) return;
    currentTabId = tab.id;

    const isForm = tab.url && tab.url.includes('docs.google.com/forms');
    if (isForm) {
      setStatus('active', 'Google Form detected');
      try {
        const resp = await chrome.tabs.sendMessage(tab.id, { action: 'getFormInfo' });
        if (resp?.success) {
          showFormDetected(resp.title, resp.questionCount);
        } else {
          showFormDetected(tab.title || 'Google Form', '?');
        }
      } catch {
        showFormDetected(tab.title || 'Google Form', '?');
      }
    } else {
      setStatus('', 'Not on a Google Form');
      showNoForm();
    }
  } catch {
    setStatus('error', 'Cannot access tab');
    showNoForm();
  }
}

function showFormDetected(title, count) {
  $('#form-detected').style.display = '';
  $('#no-form-detected').style.display = 'none';
  $('#detected-form-title').textContent = title || 'Untitled Form';
  $('#detected-form-meta').textContent = typeof count === 'number' ? `${count} questions detected` : 'Questions detected';

  // Fill button
  $('#btn-fill-form').addEventListener('click', () => fillCurrentForm());
}

function showNoForm() {
  $('#form-detected').style.display = 'none';
  $('#no-form-detected').style.display = '';
}

function setStatus(state, text) {
  const dot = $('#status-dot');
  const label = $('#status-text');
  dot.className = 'status-dot';
  if (state) dot.classList.add(state);
  label.textContent = text;
}

// ---- Fill current form ----
async function fillCurrentForm() {
  if (!currentTabId) {
    showToast('No active tab found', 'error');
    return;
  }

  const autoSubmit = $('#toggle-autosubmit')?.checked || false;

  try {
    const resp = await chrome.runtime.sendMessage({
      action: 'startSolving',
      tabId: currentTabId,
      autoSubmit,
    });

    if (resp?.started) {
      showProgress('Solving form...', 'Starting...');
      startPolling();
    } else {
      showToast(resp?.reason || 'Could not start solving', 'error');
    }
  } catch (err) {
    showToast('Error: ' + err.message, 'error');
  }
}

// ---- Link Tab ----
function initLinkTab() {
  const linkInput = $('#link-input');
  const solveBtn = $('#btn-solve-link');
  const clearBtn = $('#btn-clear-link');

  linkInput.addEventListener('input', () => {
    const val = linkInput.value.trim();
    const valid = val.startsWith('https://docs.google.com/forms');
    solveBtn.disabled = !valid;
    clearBtn.style.display = val ? '' : 'none';
  });

  clearBtn.addEventListener('click', () => {
    linkInput.value = '';
    solveBtn.disabled = true;
    clearBtn.style.display = 'none';
  });

  solveBtn.addEventListener('click', () => solveByLink());
}

async function solveByLink() {
  const url = $('#link-input').value.trim();
  if (!url) return;

  const autoSubmit = $('#toggle-link-autosubmit')?.checked ?? true;

  try {
    const resp = await chrome.runtime.sendMessage({
      action: 'startSolvingByLink',
      url,
      autoSubmit,
    });

    if (resp?.started) {
      showProgress('Opening form...', 'Working in background...');
      startPolling();
    } else {
      showToast(resp?.reason || 'Could not start', 'error');
    }
  } catch (err) {
    showToast('Error: ' + err.message, 'error');
  }
}

// =============================================
// PROGRESS OVERLAY
// =============================================
function showProgress(title, detail) {
  const overlay = $('#progress-overlay');
  overlay.style.display = '';
  $('#progress-title').textContent = title;
  $('#progress-detail').textContent = detail;
  $('#progress-bar').style.width = '0%';
  $('#progress-count').textContent = '';

  $('#btn-cancel-solve').onclick = cancelSolve;
}

function updateProgress(state) {
  if (!state) return;
  const { title, detail, current, total } = state.progress || {};
  if (title) $('#progress-title').textContent = title;
  if (detail) $('#progress-detail').textContent = detail;

  if (total > 0) {
    const pct = Math.min(Math.round((current / total) * 100), 100);
    $('#progress-bar').style.width = pct + '%';
    $('#progress-count').textContent = `${current} / ${total}`;

    // Update ring arc
    const arc = document.querySelector('.progress-ring-arc');
    if (arc) {
      const circumference = 2 * Math.PI * 30; // r=30
      const offset = circumference - (pct / 100) * circumference;
      arc.style.strokeDashoffset = offset;
    }
  }
}

function hideProgress() {
  const overlay = $('#progress-overlay');
  overlay.style.display = 'none';
}

async function cancelSolve() {
  try {
    await chrome.runtime.sendMessage({ action: 'cancelSolving' });
  } catch {}
  stopPolling();
  hideProgress();
  showToast('Solving cancelled', 'error');
}

// ---- Polling ----
function startPolling() {
  stopPolling();
  pollTimer = setInterval(async () => {
    try {
      const state = await chrome.runtime.sendMessage({ action: 'getSolveStatus' });
      updateProgress(state);

      if (state.completed) {
        stopPolling();
        hideProgress();
        await chrome.runtime.sendMessage({ action: 'clearSolveResult' });

        if (state.result?.success) {
          const { filled, total } = state.result;
          showToast(`Filled ${filled}/${total} answers!`, 'success');
          setStatus('active', 'Form solved successfully');
        } else {
          showToast(state.result?.error || 'Solve failed', 'error');
          setStatus('error', 'Solve failed');
        }
      }
    } catch {
      // popup may be closing, ignore
    }
  }, 600);
}

function stopPolling() {
  if (pollTimer) {
    clearInterval(pollTimer);
    pollTimer = null;
  }
}

function checkOngoingSolve() {
  chrome.runtime.sendMessage({ action: 'getSolveStatus' }, (state) => {
    if (chrome.runtime.lastError) return;
    if (!state) return;

    if (state.active) {
      showProgress(state.progress?.title || 'Solving...', state.progress?.detail || 'Working...');
      updateProgress(state);
      startPolling();
    } else if (state.completed && state.result) {
      if (state.result.success) {
        showToast(`Filled ${state.result.filled}/${state.result.total} answers!`, 'success');
      } else {
        showToast(state.result.error || 'Solve failed', 'error');
      }
      chrome.runtime.sendMessage({ action: 'clearSolveResult' });
    }
  });
}

// =============================================
// UPDATE CHECKER
// =============================================

async function checkAndShowUpdateBanner() {
  try {
    // First check stored update info
    const resp = await chrome.runtime.sendMessage({ action: 'getUpdateInfo' });
    if (resp?.updateAvailable && resp.updateVersion) {
      // Check if user dismissed this specific version
      if (resp.updateDismissedVersion === resp.updateVersion) {
        return; // User already dismissed this version
      }
      showUpdateBanner(resp.updateVersion);
    } else {
      // Trigger a fresh check in background
      const freshCheck = await chrome.runtime.sendMessage({ action: 'checkForUpdate' });
      if (freshCheck?.hasUpdate) {
        showUpdateBanner(freshCheck.latestVersion);
      }
    }
  } catch (e) {
    // Silently fail — update check is non-critical
    console.log('[Solvo] Update check in popup failed:', e.message);
  }
}

function showUpdateBanner(version) {
  const banner = $('#update-banner');
  if (!banner) return;

  // Set version text
  const versionText = $('#update-version-text');
  if (versionText) versionText.textContent = `v${version} is ready`;

  banner.style.display = '';

  // Update button — opens update page
  const updateBtn = $('#btn-update');
  if (updateBtn) {
    updateBtn.addEventListener('click', async () => {
      try {
        await chrome.runtime.sendMessage({ action: 'openUpdatePage' });
      } catch (e) {
        // Fallback: open website directly
        chrome.tabs.create({ url: 'https://solvo.pages.dev/download' });
      }
    });
  }

  // Dismiss button
  const dismissBtn = $('#btn-dismiss-update');
  if (dismissBtn) {
    dismissBtn.addEventListener('click', async () => {
      banner.style.display = 'none';
      try {
        await chrome.runtime.sendMessage({ action: 'dismissUpdate' });
        // Clear badge
        chrome.action.setBadgeText({ text: '' });
      } catch (e) {}
    });
  }
}

// =============================================
// TOAST
// =============================================
function showToast(msg, type = 'info') {
  const toast = $('#toast');
  const icon = $('#toast-icon');
  const msgSpan = $('#toast-msg');

  toast.className = 'toast' + (type ? ' ' + type : '');
  msgSpan.textContent = msg;

  // Set icon
  if (type === 'success') {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><polyline points="20 6 9 17 4 12"/></svg>';
  } else if (type === 'error') {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><line x1="18" y1="6" x2="6" y2="18"/><line x1="6" y1="6" x2="18" y2="18"/></svg>';
  } else {
    icon.innerHTML = '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5"><circle cx="12" cy="12" r="10"/><path d="M12 16v-4"/><path d="M12 8h.01"/></svg>';
  }

  toast.style.display = '';

  if (toastTimer) clearTimeout(toastTimer);
  toastTimer = setTimeout(() => {
    toast.style.display = 'none';
  }, 3500);
}
