// =============================================
// SOLVO — Content Script for Google Forms
// Parses questions, receives AI answers,
// and fills the form automatically.
// =============================================

(() => {
  'use strict';

  // ---- Message Listener ----
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    switch (msg.action) {
      case 'getFormInfo':
        // Use async with wait for form load
        getFormInfoAsync().then(sendResponse);
        return true;

      case 'startFilling':
        startFilling(msg.autoSubmit).then(sendResponse);
        return true; // async

      case 'extractQuestions':
        extractQuestionsForPopup().then(sendResponse);
        return true;

      case 'fillWithAnswers':
        fillWithProvidedAnswers(msg.answers, msg.autoSubmit).then(sendResponse);
        return true;
    }
  });

  // =============================================
  // FORM INFO — Quick scan for popup display
  // =============================================

  async function getFormInfoAsync() {
    let questions = parseQuestions();
    if (questions.length === 0) {
      await waitForFormLoad(8, 500);
      questions = parseQuestions();
    }
    const title = getFormTitle();
    return {
      found: questions.length > 0,
      title,
      questionCount: questions.length,
    };
  }

  function getFormInfo() {
    const title = getFormTitle();
    const questions = parseQuestions();
    return {
      found: questions.length > 0,
      title,
      questionCount: questions.length,
    };
  }

  function getFormTitle() {
    // Google Forms title selectors (broadened for 2024-2026 DOM)
    const selectors = [
      '[data-item-id] div[role="heading"][aria-level="1"]',
      '.freebirdFormviewerViewHeaderTitle',
      'div[role="heading"][aria-level="1"]',
      '.F9yp7e',
      'div.ahS2Le',
      'div.lrKTG',           // newer title container
      'h1[dir="auto"]',      // direct h1
      'div.o3Dpx',           // 2025+ variant
    ];
    for (const sel of selectors) {
      const el = document.querySelector(sel);
      if (el?.textContent?.trim()) return el.textContent.trim();
    }
    return document.title.replace(' - Google Forms', '').replace(' - Google Formular', '').trim() || 'Untitled Form';
  }

  // =============================================
  // QUESTION PARSER — Extract all questions
  // Multiple strategies for maximum compatibility
  // =============================================

  function parseQuestions() {
    console.log('[Solvo] Starting question extraction...');

    // Strategy A: Try DOM-based extraction first
    let questions = extractFromDOM();
    if (questions.length > 0) {
      console.log(`[Solvo] DOM extraction found ${questions.length} questions`);
      return questions;
    }

    // Strategy B: Parse data-params attributes (Google encodes question data here)
    questions = extractFromDataParams();
    if (questions.length > 0) {
      console.log(`[Solvo] data-params extraction found ${questions.length} questions`);
      return questions;
    }

    // Strategy C: Parse FB_PUBLIC_LOAD_DATA_ script variable
    questions = extractFromPublicLoadData();
    if (questions.length > 0) {
      console.log(`[Solvo] FB_PUBLIC_LOAD_DATA extraction found ${questions.length} questions`);
      return questions;
    }

    // Debug: dump DOM info so we can diagnose
    debugDOMDump();
    return [];
  }

  // ---- Strategy A: DOM-based extraction ----
  function extractFromDOM() {
    const questionBlocks = getQuestionBlocks();
    if (questionBlocks.length === 0) return [];

    const questions = [];
    questionBlocks.forEach((block, index) => {
      const question = parseQuestionBlock(block, index);
      if (question) questions.push(question);
    });
    return questions;
  }

  // Wait for form to load (Google Forms lazily renders questions)
  async function waitForFormLoad(maxRetries = 15, delay = 800) {
    for (let attempt = 0; attempt < maxRetries; attempt++) {
      const questions = parseQuestions();
      if (questions.length > 0) {
        console.log(`[Solvo] Form loaded after ${attempt + 1} attempt(s): ${questions.length} questions`);
        return questions.length;
      }
      console.log(`[Solvo] Waiting for form to load... attempt ${attempt + 1}/${maxRetries}`);
      await sleep(delay);
    }
    return 0;
  }

  function getQuestionBlocks() {
    // Multi-strategy block detection — try ALL and pick the one with most results
    const strategies = [
      // 1. role="listitem" with inputs — most reliable
      () => {
        const items = document.querySelectorAll('div[role="listitem"]');
        return Array.from(items).filter(b => hasQuestionContent(b));
      },
      // 2. data-params containers
      () => {
        const items = document.querySelectorAll('div[data-params]');
        return Array.from(items).filter(b => hasQuestionContent(b));
      },
      // 3. data-item-id containers
      () => {
        const items = document.querySelectorAll('div[data-item-id]');
        return Array.from(items).filter(b => hasQuestionContent(b));
      },
      // 4. jscontroller containers that have question-like structure
      () => {
        const items = document.querySelectorAll('div[jscontroller][data-params]');
        return Array.from(items);
      },
      // 5. Known class-based (may change frequently)
      () => {
        const selectors = [
          'div.freebirdFormviewerComponentsQuestionBaseRoot',
          'div.Qr7Oae',
          'div.geS5n',
          'div.m2Bhab',
        ];
        for (const sel of selectors) {
          const blocks = document.querySelectorAll(sel);
          if (blocks.length > 0) return Array.from(blocks);
        }
        return [];
      },
      // 6. Containers with radiogroup or group role
      () => Array.from(document.querySelectorAll('div[role="group"], div[role="radiogroup"]')),
      // 7. Walk up from inputs
      () => walkUpFromInputs(),
    ];

    for (const strategy of strategies) {
      try {
        const blocks = strategy();
        if (blocks.length > 0) {
          console.log(`[Solvo] Got ${blocks.length} question blocks`);
          return blocks;
        }
      } catch (e) {
        console.warn('[Solvo] Strategy failed:', e);
      }
    }

    return [];
  }

  function hasQuestionContent(block) {
    // Check if block has any interactive form element
    return block.querySelector(
      'input:not([type="hidden"]):not([type="submit"]):not([type="button"]), ' +
      'textarea, ' +
      'div[role="radio"], div[role="checkbox"], ' +
      'div[role="listbox"], div[role="radiogroup"], div[role="group"]'
    ) !== null;
  }

  function walkUpFromInputs() {
    const allInputs = document.querySelectorAll(
      'input:not([type="hidden"]):not([type="submit"]), textarea, div[role="radio"], div[role="checkbox"], div[role="listbox"]'
    );
    const containerSet = new Set();
    const containers = [];
    allInputs.forEach(inp => {
      let el = inp.parentElement;
      for (let i = 0; i < 8 && el; i++) {
        if (el.offsetHeight > 40 && el.children.length >= 2 && !containerSet.has(el)) {
          const inputCount = el.querySelectorAll('input, textarea, div[role="radio"], div[role="checkbox"]').length;
          if (inputCount > 0 && inputCount <= 20) {
            containerSet.add(el);
            containers.push(el);
            break;
          }
        }
        el = el.parentElement;
      }
    });
    return containers;
  }

  // ---- Strategy B: Extract from data-params ----
  function extractFromDataParams() {
    const questions = [];
    const paramsEls = document.querySelectorAll('div[data-params]');

    paramsEls.forEach((el, idx) => {
      try {
        let raw = el.getAttribute('data-params');
        if (!raw) return;

        // data-params format: %.@.[entryId,"Question text",description,type,[[optId,"Option"],...],...]
        // Clean up the leading markers
        raw = raw.replace(/^%.@\./, '');

        // Try to parse as JSON array
        let parsed;
        try {
          parsed = JSON.parse(raw);
        } catch {
          // Sometimes it's not valid JSON, try to extract text manually
          const textMatch = raw.match(/,\s*"([^"]+)"/);
          if (textMatch) {
            questions.push({
              index: idx,
              text: textMatch[1],
              description: '',
              required: false,
              type: 'short_answer',
              options: [],
              element: el,
            });
          }
          return;
        }

        if (!Array.isArray(parsed) || parsed.length < 2) return;

        const questionText = parsed[1]; // Question text is typically at index 1
        if (!questionText || typeof questionText !== 'string') return;

        const description = (typeof parsed[2] === 'string') ? parsed[2] : '';
        const typeCode = parsed[3]; // 0=short, 1=long, 2=radio, 3=dropdown, 4=checkbox, 5=linear, 7=grid, 9=date, 10=time
        const required = parsed[4] === 1 || parsed[4] === true;

        // Map Google's type codes
        let type = 'short_answer';
        switch (typeCode) {
          case 0: type = 'short_answer'; break;
          case 1: type = 'long_answer'; break;
          case 2: type = 'multiple_choice'; break;
          case 3: type = 'dropdown'; break;
          case 4: type = 'checkbox'; break;
          case 5: type = 'linear_scale'; break;
          case 7: type = 'grid'; break;
          case 9: type = 'date'; break;
          case 10: type = 'time'; break;
        }

        // Extract options (typically at index 4 as nested arrays -- [[id, "text"], ...])
        let options = [];
        // Look for an array of arrays containing option strings
        for (let i = 3; i < Math.min(parsed.length, 10); i++) {
          if (Array.isArray(parsed[i])) {
            const candidateOpts = [];
            parsed[i].forEach(item => {
              if (Array.isArray(item)) {
                // Find the string element in the sub-array
                const str = item.find(x => typeof x === 'string' && x.length > 0);
                if (str) candidateOpts.push(str);
              }
            });
            if (candidateOpts.length > 1) {
              options = candidateOpts;
              break;
            }
          }
        }

        questions.push({
          index: idx,
          text: questionText,
          description,
          required,
          type,
          options,
          element: el,
        });
      } catch (e) {
        console.warn('[Solvo] data-params parse error for element', idx, e);
      }
    });

    return questions;
  }

  // ---- Strategy C: Extract from FB_PUBLIC_LOAD_DATA_ ----
  function extractFromPublicLoadData() {
    const questions = [];
    try {
      // Find the script containing FB_PUBLIC_LOAD_DATA_
      const scripts = document.querySelectorAll('script');
      let loadData = null;

      for (const script of scripts) {
        const content = script.textContent || '';
        // Try multiple regex patterns — the data can be very large
        if (content.includes('FB_PUBLIC_LOAD_DATA_')) {
          try {
            // Extract everything between = and the final semicolon
            const startIdx = content.indexOf('FB_PUBLIC_LOAD_DATA_');
            const eqIdx = content.indexOf('=', startIdx);
            if (eqIdx > -1) {
              const jsonStr = content.substring(eqIdx + 1).replace(/;\s*$/, '').trim();
              loadData = JSON.parse(jsonStr);
            }
          } catch (e) {
            console.warn('[Solvo] FB_PUBLIC_LOAD_DATA_ JSON parse failed:', e.message);
          }
          break;
        }
      }

      // Also check if it's on window (some forms embed it directly)
      if (!loadData && typeof window.FB_PUBLIC_LOAD_DATA_ !== 'undefined') {
        loadData = window.FB_PUBLIC_LOAD_DATA_;
      }

      if (!loadData) return [];

      // FB_PUBLIC_LOAD_DATA_ structure:
      // loadData[1][1] = array of question items
      // Each item: [itemId, questionText, description, typeCode, fieldGroups, ...]
      //   fieldGroups = [[fieldId, optionsArray, required, ...], ...]
      //   optionsArray = [["OptionText", null, null, null, 0], ...] or null
      const formData = loadData?.[1]?.[1];
      if (!Array.isArray(formData)) {
        console.log('[Solvo] FB_PUBLIC_LOAD_DATA found but no questions at [1][1]');
        return [];
      }

      console.log(`[Solvo] FB_PUBLIC_LOAD_DATA has ${formData.length} items`);

      // Collect all DOM question blocks for element mapping
      const domBlocks = getDOMQuestionBlocksForMapping();
      console.log(`[Solvo] Found ${domBlocks.length} DOM blocks for element mapping`);

      formData.forEach((item, idx) => {
        if (!Array.isArray(item)) return;

        const itemId = item[0]; // e.g. 69212096
        const questionText = item[1];
        if (!questionText || typeof questionText !== 'string') return;

        const description = (typeof item[2] === 'string') ? item[2] : '';
        const typeCode = item[3];
        let type = 'short_answer';
        switch (typeCode) {
          case 0: type = 'short_answer'; break;
          case 1: type = 'long_answer'; break;
          case 2: type = 'multiple_choice'; break;
          case 3: type = 'dropdown'; break;
          case 4: type = 'checkbox'; break;
          case 5: type = 'linear_scale'; break;
          case 7: type = 'grid'; break;
          case 9: type = 'date'; break;
          case 10: type = 'time'; break;
        }

        // Extract options and required from fieldGroups at item[4]
        let options = [];
        let required = false;
        let entryFieldId = null;

        if (Array.isArray(item[4])) {
          // item[4] is an array of field groups: [[fieldId, optionsArr, required, ...], ...]
          for (const fieldGroup of item[4]) {
            if (!Array.isArray(fieldGroup)) continue;

            entryFieldId = fieldGroup[0]; // e.g. 156384672 → used as entry.156384672

            // Required flag is at fieldGroup[2]
            if (fieldGroup[2] === 1) required = true;

            // Options array is at fieldGroup[1]
            const optionsArr = fieldGroup[1];
            if (Array.isArray(optionsArr)) {
              for (const opt of optionsArr) {
                // Each option is ["OptionText", null, null, null, 0]
                if (Array.isArray(opt) && typeof opt[0] === 'string') {
                  options.push(opt[0]);
                }
              }
            }
          }
        }

        // Try to find the matching DOM element for this question
        let element = null;

        // Method 1: Find by data-item-id matching the item ID
        if (itemId) {
          element = document.querySelector(`div[data-item-id="${itemId}"]`);
        }

        // Method 2: Find by entry field ID in input name attribute
        if (!element && entryFieldId) {
          const input = document.querySelector(
            `input[name="entry.${entryFieldId}"], textarea[name="entry.${entryFieldId}"], [data-params*="${entryFieldId}"]`
          );
          if (input) {
            // Walk up to find the question container
            element = input.closest('div[role="listitem"]') ||
                      input.closest('div[data-params]') ||
                      input.closest('div[data-item-id]') ||
                      input.parentElement?.parentElement?.parentElement;
          }
        }

        // Method 3: Use positional mapping from DOM blocks
        if (!element && idx < domBlocks.length) {
          element = domBlocks[idx];
        }

        // Fallback to document.body
        if (!element) element = document.body;

        console.log(`[Solvo] Q${idx}: "${questionText.substring(0, 50)}..." type=${type} options=${options.length} required=${required} hasElement=${element !== document.body}`);

        questions.push({
          index: idx,
          text: questionText.trim(),
          description,
          required,
          type,
          options,
          element,
          entryFieldId, // store for direct filling
          itemId,       // store for DOM lookup
        });
      });
    } catch (e) {
      console.warn('[Solvo] FB_PUBLIC_LOAD_DATA parse error:', e);
    }
    return questions;
  }

  // Get DOM question blocks in order for positional mapping
  function getDOMQuestionBlocksForMapping() {
    // Try multiple strategies to find question containers in DOM order
    const strategies = [
      () => Array.from(document.querySelectorAll('div[data-item-id]')),
      () => Array.from(document.querySelectorAll('div[role="listitem"]')).filter(b => hasQuestionContent(b)),
      () => Array.from(document.querySelectorAll('div[data-params]')).filter(b => hasQuestionContent(b)),
      () => Array.from(document.querySelectorAll('div.Qr7Oae, div.geS5n, div.m2Bhab')),
    ];
    for (const strategy of strategies) {
      const blocks = strategy();
      if (blocks.length > 0) return blocks;
    }
    return [];
  }

  // ---- Debug: log DOM info when extraction fails ----
  function debugDOMDump() {
    console.log('[Solvo] === DEBUG DOM DUMP (extraction failed) ===');
    console.log('[Solvo] URL:', window.location.href);
    console.log('[Solvo] Title:', document.title);
    console.log('[Solvo] div[role="listitem"] count:', document.querySelectorAll('div[role="listitem"]').length);
    console.log('[Solvo] div[data-params] count:', document.querySelectorAll('div[data-params]').length);
    console.log('[Solvo] div[data-item-id] count:', document.querySelectorAll('div[data-item-id]').length);
    console.log('[Solvo] div[role="radio"] count:', document.querySelectorAll('div[role="radio"]').length);
    console.log('[Solvo] div[role="checkbox"] count:', document.querySelectorAll('div[role="checkbox"]').length);
    console.log('[Solvo] input count:', document.querySelectorAll('input').length);
    console.log('[Solvo] textarea count:', document.querySelectorAll('textarea').length);
    console.log('[Solvo] div[role="group"] count:', document.querySelectorAll('div[role="group"]').length);
    console.log('[Solvo] div[role="radiogroup"] count:', document.querySelectorAll('div[role="radiogroup"]').length);
    console.log('[Solvo] form count:', document.querySelectorAll('form').length);
    console.log('[Solvo] div[jscontroller] count:', document.querySelectorAll('div[jscontroller]').length);
    // Log first 2000 chars of body HTML for class analysis
    const bodySnippet = document.body?.innerHTML?.substring(0, 2000) || 'EMPTY';
    console.log('[Solvo] Body snippet:', bodySnippet);
    console.log('[Solvo] === END DEBUG ===');
  }

  function parseQuestionBlock(block, index) {
    // Try getting text from data-params first
    const dpText = getTextFromDataParams(block);
    const text = dpText || getQuestionText(block);
    if (!text) return null;

    const description = getQuestionDescription(block);
    const required = isRequired(block);
    const type = detectQuestionType(block);
    const options = getQuestionOptions(block, type);

    return {
      index,
      text,
      description,
      required,
      type,
      options,
      element: block,
    };
  }

  function getTextFromDataParams(block) {
    // Try to get question text from data-params attribute on the block itself or a child
    const dpEl = block.hasAttribute('data-params') ? block : block.querySelector('[data-params]');
    if (!dpEl) return null;

    try {
      let raw = dpEl.getAttribute('data-params');
      raw = raw.replace(/^%.@\./, '');
      const parsed = JSON.parse(raw);
      if (Array.isArray(parsed) && typeof parsed[1] === 'string') {
        return parsed[1];
      }
    } catch {}

    // Fallback: extract first quoted string from attribute
    const match = dpEl.getAttribute('data-params')?.match(/,\s*"([^"]{3,})"/);
    return match ? match[1] : null;
  }

  function getQuestionText(block) {
    // Question title selectors (ordered by specificity, broadened for 2024-2026)
    const selectors = [
      'div[role="heading"] span.M7eMe',
      'div[role="heading"] span',
      'div[role="heading"]',
      '.freebirdFormviewerComponentsQuestionBaseTitle',
      'span.M7eMe',
      'div[data-initial-value]',
      '[data-initial-value]',
      'span.HoXoMd',
      'div.HoXoMd',
    ];

    for (const sel of selectors) {
      const el = block.querySelector(sel);
      if (el?.textContent?.trim()) {
        let text = el.textContent.trim();
        // Remove asterisk / required marker
        text = text.replace(/\s*\*\s*$/, '');
        if (text.length > 0) return text;
      }
    }

    // Fallback: try to extract the first substantial text node before inputs
    const allText = [];
    const walker = document.createTreeWalker(block, NodeFilter.SHOW_TEXT, null);
    let node;
    while ((node = walker.nextNode())) {
      const t = node.textContent.trim();
      if (t.length > 2 && !node.parentElement?.closest('input, textarea, [role="radio"], [role="checkbox"], [role="listbox"]')) {
        allText.push(t);
      }
    }
    // Return the first meaningful text that looks like a question
    for (const t of allText) {
      if (t.length > 2 && t !== '*' && t !== 'Required') {
        return t.replace(/\s*\*\s*$/, '');
      }
    }

    return null;
  }

  function getQuestionDescription(block) {
    const selectors = [
      '.freebirdFormviewerComponentsQuestionBaseDescription',
      'div[dir="auto"].ulDsOb',
      '.HoXoMd',
      'div.gubaDc',
    ];
    for (const sel of selectors) {
      const el = block.querySelector(sel);
      if (el?.textContent?.trim()) return el.textContent.trim();
    }
    return '';
  }

  function isRequired(block) {
    if (block.querySelector('[aria-required="true"], [data-required="true"], .freebirdFormviewerComponentsQuestionBaseRequiredAsterisk')) return true;
    // Check for asterisk in heading/title area only (not entire block)
    const heading = block.querySelector('div[role="heading"], span.M7eMe');
    if (heading && heading.textContent.includes('*')) return true;
    return false;
  }

  function detectQuestionType(block) {
    if (block.querySelector('div[role="radiogroup"], div[role="radio"]')) return 'multiple_choice';
    if (block.querySelector('div[role="checkbox"], div[role="group"] div[role="checkbox"]')) return 'checkbox';
    if (block.querySelector('div[role="listbox"]')) return 'dropdown';
    if (block.querySelector('textarea')) return 'long_answer';
    if (block.querySelector('input[type="date"]')) return 'date';
    if (block.querySelector('input[type="time"]')) return 'time';
    if (block.querySelector('input[type="number"]')) return 'number';

    // Linear scale detection: multiple radios in a row with numeric labels
    const radios = block.querySelectorAll('div[role="radio"]');
    if (radios.length >= 3 && radios.length <= 11) {
      const labels = Array.from(radios).map(r => r.getAttribute('aria-label') || r.textContent.trim());
      if (labels.every(l => /^\d+$/.test(l))) return 'linear_scale';
    }

    if (block.querySelector('input[type="text"]')) return 'short_answer';
    return 'short_answer';
  }

  function getQuestionOptions(block, type) {
    const options = [];

    if (type === 'multiple_choice' || type === 'linear_scale') {
      block.querySelectorAll('div[role="radio"]').forEach(radio => {
        const label = getOptionLabel(radio);
        if (label) options.push(label);
      });
    } else if (type === 'checkbox') {
      block.querySelectorAll('div[role="checkbox"]').forEach(cb => {
        const label = getOptionLabel(cb);
        if (label) options.push(label);
      });
    } else if (type === 'dropdown') {
      block.querySelectorAll('div[role="option"]').forEach(opt => {
        const label = opt.textContent.trim();
        if (label && label !== 'Choose') options.push(label);
      });
    }

    return options;
  }

  function getOptionLabel(el) {
    // Try aria-label first
    const ariaLabel = el.getAttribute('aria-label');
    if (ariaLabel) return ariaLabel.trim();

    // Try data-value
    const dataValue = el.getAttribute('data-value');
    if (dataValue) return dataValue.trim();

    // Try data-answer-value
    const answerValue = el.getAttribute('data-answer-value');
    if (answerValue) return answerValue.trim();

    // Try label text inside the option (span, label, div with text)
    const labelSelectors = ['span.aDTYNe', 'span.docssharedWizToggleLabeledLabelText', 'span', 'label'];
    for (const sel of labelSelectors) {
      const labelEl = el.querySelector(sel);
      const t = labelEl?.textContent?.trim();
      if (t && t.length > 0) return t;
    }

    // Fallback to own text
    const text = el.textContent.trim();
    return text || null;
  }

  // =============================================
  // PERSONAL QUESTION DETECTION
  // =============================================

  const PERSONAL_PATTERNS = {
    name: /\b(your name|full name|first name|last name|student name|candidate name|participant name|respondent name)\b/i,
    email: /\b(email|e-mail|mail id|email address|your email|email id)\b/i,
    phone: /\b(phone|mobile|cell|telephone|contact number|phone number|mobile number|whatsapp)\b/i,
    organization: /\b(school|college|university|organization|institution|company|department|branch|class|section|semester|year)\b/i,
    studentId: /\b(roll|enroll|registration|student id|prn|usn|reg no|admission|id number|roll no|roll number)\b/i,
    address: /\b(address|city|state|pin|zip|postal|location|town|district|country)\b/i,
  };

  function detectPersonalField(questionText) {
    const lower = questionText.toLowerCase();
    for (const [field, pattern] of Object.entries(PERSONAL_PATTERNS)) {
      if (pattern.test(lower)) return field;
    }
    return null;
  }

  // =============================================
  // FORM FILLER — Apply answers to the form
  // =============================================

  async function startFilling(autoSubmit) {
    try {
      // Show floating indicator
      showSolvoOverlay('Analyzing questions...');

      // Parse all questions
      const questions = parseQuestions();
      if (questions.length === 0) {
        hideSolvoOverlay();
        return { success: false, error: 'No questions found on this page' };
      }

      // Get profile
      const profile = await sendMessage({ action: 'getProfile' });

      // Separate personal and quiz questions
      const personalAnswers = {};
      const quizQuestions = [];

      questions.forEach((q, i) => {
        const personalField = detectPersonalField(q.text);
        if (personalField && profile?.[personalField]) {
          personalAnswers[i] = {
            questionIndex: i,
            answer: profile[personalField],
            type: 'text',
          };
        } else {
          quizQuestions.push({
            index: i,
            text: q.text,
            description: q.description,
            type: q.type,
            options: q.options,
            required: q.required,
          });
        }
      });

      updateSolvoOverlay(`Solving ${quizQuestions.length} questions with AI...`);

      // Send quiz questions to AI via background (Gemini with model fallback)
      let aiAnswers = [];
      if (quizQuestions.length > 0) {
        const aiResult = await sendMessage({
          action: 'queryAI',
          questions: quizQuestions,
          profile,
        });

        if (!aiResult?.success) {
          hideSolvoOverlay();
          return { success: false, error: aiResult?.error || 'AI query failed', needsFallback: aiResult?.needsFallback };
        }

        aiAnswers = aiResult.answers || [];
      }

      // Merge answers — map AI answers back to original indices
      const allAnswers = { ...personalAnswers };
      aiAnswers.forEach(a => {
        // Find the original index from quiz questions
        const quizQ = quizQuestions[a.questionIndex];
        if (quizQ) {
          allAnswers[quizQ.index] = a;
        }
      });

      // Fill each question
      let filled = 0;
      for (let i = 0; i < questions.length; i++) {
        const q = questions[i];
        const answer = allAnswers[i];
        if (answer) {
          updateSolvoOverlay(`Filling question ${i + 1} of ${questions.length}...`);

          // Send progress to popup
          try {
            chrome.runtime.sendMessage({
              action: 'progressUpdate',
              title: 'Filling answers...',
              detail: `Question ${i + 1}: ${q.text.substring(0, 40)}...`,
              current: i + 1,
              total: questions.length,
            });
          } catch {}

          await fillQuestion(q, answer);
          filled++;
          // Human-like delay between questions (1-3 seconds, with occasional longer pauses)
          const baseDelay = rand(800, 2200);
          const extraPause = Math.random() < 0.15 ? rand(1500, 3000) : 0; // 15% chance of "reading" pause
          await sleep(baseDelay + extraPause);
        }
      }

      // Handle multi-page forms
      const nextBtn = findNextButton();
      if (nextBtn) {
        updateSolvoOverlay('Moving to next page...');
        nextBtn.click();
        await sleep(2000);
        // Recursively fill next page
        const nextResult = await startFilling(autoSubmit);
        return nextResult;
      }

      // Auto-submit if enabled
      if (autoSubmit) {
        updateSolvoOverlay('Submitting form...');
        await sleep(1000);
        const submitBtn = findSubmitButton();
        if (submitBtn) {
          submitBtn.click();
        }
      }

      hideSolvoOverlay();
      showSolvoComplete(filled, questions.length);

      return { success: true, filled, total: questions.length };
    } catch (err) {
      hideSolvoOverlay();
      return { success: false, error: err.message };
    }
  }

  // =============================================
  // POPUP-DRIVEN FLOW — extract & fill separately
  // =============================================

  async function extractQuestionsForPopup() {
    try {
      // Wait for form to fully render before extraction
      let questions = parseQuestions();
      if (questions.length === 0) {
        console.log('[Solvo] No questions found initially, waiting for form to load...');
        await waitForFormLoad(15, 800);
        questions = parseQuestions();
      }

      if (questions.length === 0) {
        // Log extensive debug info
        debugDOMDump();
        const dpCount = document.querySelectorAll('div[data-params]').length;
        const inputCount = document.querySelectorAll('input').length;
        return {
          success: false,
          error: `No questions found. DOM has ${dpCount} data-params elements and ${inputCount} inputs. Open Console (F12) for debug info. The form may still be loading or requires interaction first.`
        };
      }

      const profile = await sendMessage({ action: 'getProfile' });
      const personalAnswers = {};
      const quizQuestions = [];

      questions.forEach((q, i) => {
        const personalField = detectPersonalField(q.text);
        if (personalField && profile?.[personalField]) {
          personalAnswers[i] = {
            questionIndex: i,
            answer: profile[personalField],
            type: 'text',
          };
        } else {
          quizQuestions.push({
            index: i,
            text: q.text,
            description: q.description,
            type: q.type,
            options: q.options,
            required: q.required,
          });
        }
      });

      console.log(`[Solvo] Extraction complete: ${quizQuestions.length} quiz + ${Object.keys(personalAnswers).length} personal`);
      return {
        success: true,
        title: getFormTitle(),
        totalQuestions: questions.length,
        quizQuestions,
        personalAnswers,
      };
    } catch (err) {
      console.error('[Solvo] extractQuestionsForPopup error:', err);
      return { success: false, error: err.message };
    }
  }

  async function fillWithProvidedAnswers(allAnswersArr, autoSubmit) {
    try {
      showSolvoOverlay('Filling answers...');
      const questions = parseQuestions();

      // Convert array to map by question index
      const allAnswers = {};
      allAnswersArr.forEach(a => {
        if (a.questionIndex !== undefined) {
          allAnswers[a.questionIndex] = a;
        }
      });

      let filled = 0;
      for (let i = 0; i < questions.length; i++) {
        const q = questions[i];
        const answer = allAnswers[i];
        if (answer) {
          updateSolvoOverlay(`Filling question ${i + 1} of ${questions.length}...`);
          try {
            chrome.runtime.sendMessage({
              action: 'progressUpdate',
              title: 'Filling answers...',
              detail: `Question ${i + 1}: ${q.text.substring(0, 40)}...`,
              current: i + 1,
              total: questions.length,
            });
          } catch {}
          await fillQuestion(q, answer);
          filled++;
          // Human-like delay between questions
          const d = rand(800, 2200);
          const p = Math.random() < 0.15 ? rand(1500, 3000) : 0;
          await sleep(d + p);
        }
      }

      // Handle multi-page forms
      const nextBtn = findNextButton();
      if (nextBtn) {
        updateSolvoOverlay('Moving to next page...');
        nextBtn.click();
        await sleep(2000);
        // Signal popup to re-extract and fill next page
        hideSolvoOverlay();
        return { success: true, filled, total: questions.length, hasNextPage: true };
      }

      if (autoSubmit) {
        updateSolvoOverlay('Submitting form...');
        await sleep(1000);
        const submitBtn = findSubmitButton();
        if (submitBtn) submitBtn.click();
      }

      hideSolvoOverlay();
      showSolvoComplete(filled, questions.length);
      return { success: true, filled, total: questions.length };
    } catch (err) {
      hideSolvoOverlay();
      return { success: false, error: err.message };
    }
  }

  async function fillQuestion(question, answer) {
    let block = question.element;
    const answerValue = answer.answer;

    // If element is document.body (FB_PUBLIC_LOAD_DATA_ fallback), try to find the actual block
    if (block === document.body) {
      // Try to find by entry field ID
      if (question.entryFieldId) {
        const input = document.querySelector(
          `input[name="entry.${question.entryFieldId}"], textarea[name="entry.${question.entryFieldId}"]`
        );
        if (input) {
          block = input.closest('div[role="listitem"]') ||
                  input.closest('div[data-params]') ||
                  input.closest('div[data-item-id]') ||
                  input.parentElement?.parentElement?.parentElement || block;
        }
      }
      // Try by item ID
      if (block === document.body && question.itemId) {
        const el = document.querySelector(`div[data-item-id="${question.itemId}"]`);
        if (el) block = el;
      }
      // Try positional fallback
      if (block === document.body) {
        const domBlocks = getDOMQuestionBlocksForMapping();
        if (question.index < domBlocks.length) {
          block = domBlocks[question.index];
        }
      }
    }

    switch (question.type) {
      case 'short_answer':
      case 'number':
        await fillTextInput(block, answerValue);
        break;

      case 'long_answer':
        await fillTextArea(block, answerValue);
        break;

      case 'multiple_choice':
      case 'linear_scale':
        await selectRadio(block, answerValue);
        break;

      case 'checkbox':
        await selectCheckboxes(block, answerValue);
        break;

      case 'dropdown':
        await selectDropdown(block, answerValue);
        break;

      case 'date':
        await fillDateInput(block, answerValue);
        break;

      case 'time':
        await fillTimeInput(block, answerValue);
        break;

      default:
        await fillTextInput(block, answerValue);
    }
  }

  // =============================================
  // HUMANIZED INPUT FILLERS
  // Natural typing, scrolling, realistic delays
  // =============================================

  // Random number between min and max
  function rand(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
  }

  // Humanized sleep — slightly randomized
  function humanDelay(base = 100, variance = 80) {
    return sleep(base + Math.random() * variance);
  }

  // Scroll element into view smoothly like a human
  function scrollToElement(el) {
    el.scrollIntoView({ behavior: 'smooth', block: 'center' });
  }

  // Simulate mouse movement to element (dispatch pointer events)
  function simulateMouseInteraction(el) {
    const rect = el.getBoundingClientRect();
    const x = rect.left + rand(5, Math.max(10, rect.width - 5));
    const y = rect.top + rand(3, Math.max(6, rect.height - 3));

    el.dispatchEvent(new PointerEvent('pointerover', { bubbles: true, clientX: x, clientY: y }));
    el.dispatchEvent(new MouseEvent('mouseover', { bubbles: true, clientX: x, clientY: y }));
    el.dispatchEvent(new MouseEvent('mousemove', { bubbles: true, clientX: x, clientY: y }));
  }

  // Humanized character-by-character typing
  async function humanType(input, value) {
    const text = String(value);
    scrollToElement(input);
    await humanDelay(200, 300); // pause before starting

    // Simulate mouse hover then click
    simulateMouseInteraction(input);
    await humanDelay(80, 120);

    input.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
    await humanDelay(30, 50);
    input.focus();
    input.dispatchEvent(new FocusEvent('focus', { bubbles: true }));
    input.dispatchEvent(new MouseEvent('mouseup', { bubbles: true }));
    input.dispatchEvent(new MouseEvent('click', { bubbles: true }));
    await humanDelay(100, 200);

    // Clear existing value
    const setter = input.tagName === 'TEXTAREA'
      ? Object.getOwnPropertyDescriptor(window.HTMLTextAreaElement.prototype, 'value')?.set
      : Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, 'value')?.set;

    if (setter) setter.call(input, '');
    else input.value = '';
    input.dispatchEvent(new Event('input', { bubbles: true }));

    // Type character by character
    let current = '';
    for (let i = 0; i < text.length; i++) {
      const char = text[i];
      current += char;

      // Dispatch keydown
      input.dispatchEvent(new KeyboardEvent('keydown', {
        key: char, code: `Key${char.toUpperCase()}`,
        bubbles: true, cancelable: true,
      }));

      // Set value
      if (setter) setter.call(input, current);
      else input.value = current;

      // Dispatch input event
      input.dispatchEvent(new Event('input', { bubbles: true }));

      // Dispatch keyup
      input.dispatchEvent(new KeyboardEvent('keyup', {
        key: char, code: `Key${char.toUpperCase()}`,
        bubbles: true, cancelable: true,
      }));

      // Human typing speed: 30-120ms per character with occasional pauses
      if (char === ' ' || char === '.' || char === ',') {
        await sleep(rand(80, 200)); // longer pause at word boundaries
      } else {
        await sleep(rand(25, 90)); // normal typing speed
      }

      // Occasional longer "thinking" pause (5% chance)
      if (Math.random() < 0.05 && text.length > 10) {
        await sleep(rand(200, 500));
      }
    }

    // Final events after typing completes
    await humanDelay(100, 200);
    input.dispatchEvent(new Event('change', { bubbles: true }));
    await humanDelay(50, 100);
    input.dispatchEvent(new FocusEvent('blur', { bubbles: true }));
  }

  // Humanized click on a radio/checkbox option
  async function humanClick(el) {
    scrollToElement(el);
    await humanDelay(150, 250); // pause before clicking

    simulateMouseInteraction(el);
    await humanDelay(60, 120);

    el.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
    await humanDelay(30, 80);
    el.click();
    el.dispatchEvent(new MouseEvent('mouseup', { bubbles: true }));
    await humanDelay(50, 100);
  }

  // ---- Input Fillers (humanized) ----

  async function fillTextInput(block, value) {
    const input = block.querySelector('input[type="text"], input[type="number"], input:not([type="hidden"])');
    if (!input) return;
    await humanType(input, String(value));
  }

  async function fillTextArea(block, value) {
    const textarea = block.querySelector('textarea');
    if (!textarea) return;
    await humanType(textarea, String(value));
  }

  async function selectRadio(block, value) {
    const radios = block.querySelectorAll('div[role="radio"]');
    const valueStr = String(value).trim().toLowerCase();

    for (const radio of radios) {
      const label = getOptionLabel(radio)?.toLowerCase();
      if (label && (label === valueStr || label.includes(valueStr) || valueStr.includes(label))) {
        radio.click();
        await sleep(100);
        return;
      }
    }

    // Fuzzy match: find closest option
    let bestMatch = null;
    let bestScore = 0;
    for (const radio of radios) {
      const label = getOptionLabel(radio);
      if (label) {
        const score = similarity(label.toLowerCase(), valueStr);
        if (score > bestScore) {
          bestScore = score;
          bestMatch = radio;
        }
      }
    }
    if (bestMatch && bestScore > 0.3) {
      bestMatch.click();
    }
  }

  async function selectCheckboxes(block, value) {
    const checkboxes = block.querySelectorAll('div[role="checkbox"]');
    const values = Array.isArray(value)
      ? value.map(v => String(v).trim().toLowerCase())
      : String(value).split('\n').map(v => v.trim().toLowerCase()).filter(Boolean);

    for (const cb of checkboxes) {
      const label = getOptionLabel(cb)?.toLowerCase();
      if (label && values.some(v => label === v || label.includes(v) || v.includes(label))) {
        const isChecked = cb.getAttribute('aria-checked') === 'true';
        if (!isChecked) {
          cb.click();
          await sleep(100);
        }
      }
    }
  }

  async function selectDropdown(block, value) {
    const listbox = block.querySelector('div[role="listbox"]');
    if (!listbox) return;

    // Open dropdown
    listbox.click();
    await sleep(500);

    const valueStr = String(value).trim().toLowerCase();
    const options = document.querySelectorAll('div[role="option"], div[data-value]');

    for (const opt of options) {
      const text = opt.textContent.trim().toLowerCase();
      if (text === valueStr || text.includes(valueStr) || valueStr.includes(text)) {
        opt.click();
        await sleep(200);
        return;
      }
    }

    // Fuzzy match
    let bestMatch = null;
    let bestScore = 0;
    for (const opt of options) {
      const text = opt.textContent.trim();
      if (text && text !== 'Choose') {
        const score = similarity(text.toLowerCase(), valueStr);
        if (score > bestScore) {
          bestScore = score;
          bestMatch = opt;
        }
      }
    }
    if (bestMatch && bestScore > 0.3) {
      bestMatch.click();
    }
  }

  async function fillDateInput(block, value) {
    const inputs = block.querySelectorAll('input[type="text"], input[type="date"]');
    if (inputs.length === 0) return;

    if (inputs.length === 1 && inputs[0].type === 'date') {
      simulateTyping(inputs[0], value);
    } else {
      // Multiple inputs: day, month, year
      const dateStr = String(value);
      const parts = dateStr.split(/[-/]/);
      inputs.forEach((inp, i) => {
        if (parts[i]) simulateTyping(inp, parts[i]);
      });
    }
  }

  async function fillTimeInput(block, value) {
    const inputs = block.querySelectorAll('input[type="text"], input[type="time"]');
    if (inputs.length === 0) return;

    if (inputs.length === 1) {
      simulateTyping(inputs[0], value);
    } else {
      const timeStr = String(value);
      const parts = timeStr.split(':');
      inputs.forEach((inp, i) => {
        if (parts[i]) simulateTyping(inp, parts[i]);
      });
    }
  }

  // ---- Navigation ----

  function findSubmitButton() {
    const selectors = [
      'div[role="button"][jsname="M2UYVd"]',
      'div[role="button"] span',
      'button[type="submit"]',
    ];

    // Search for submit-like buttons
    const allButtons = document.querySelectorAll('div[role="button"], button');
    for (const btn of allButtons) {
      const text = btn.textContent.trim().toLowerCase();
      if (text === 'submit' || text === 'send' || text === 'отправить' || text === 'enviar' || text === 'soumettre') {
        return btn;
      }
    }

    return null;
  }

  function findNextButton() {
    const allButtons = document.querySelectorAll('div[role="button"], button');
    for (const btn of allButtons) {
      const text = btn.textContent.trim().toLowerCase();
      if (text === 'next' || text === 'continue' || text === 'siguiente' || text === 'suivant' || text === 'далее') {
        return btn;
      }
    }
    return null;
  }

  // ---- String Similarity (Levenshtein-based) ----
  function similarity(s1, s2) {
    if (s1 === s2) return 1;
    const len1 = s1.length;
    const len2 = s2.length;
    if (len1 === 0 || len2 === 0) return 0;

    const matrix = [];
    for (let i = 0; i <= len1; i++) matrix[i] = [i];
    for (let j = 0; j <= len2; j++) matrix[0][j] = j;

    for (let i = 1; i <= len1; i++) {
      for (let j = 1; j <= len2; j++) {
        const cost = s1[i - 1] === s2[j - 1] ? 0 : 1;
        matrix[i][j] = Math.min(
          matrix[i - 1][j] + 1,
          matrix[i][j - 1] + 1,
          matrix[i - 1][j - 1] + cost
        );
      }
    }

    const maxLen = Math.max(len1, len2);
    return 1 - matrix[len1][len2] / maxLen;
  }

  // ---- Messaging Helper ----
  function sendMessage(msg) {
    return new Promise(resolve => {
      chrome.runtime.sendMessage(msg, resolve);
    });
  }

  function sleep(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  // =============================================
  // FLOATING UI — Overlay indicators on the form
  // =============================================

  let overlayEl = null;

  function showSolvoOverlay(text) {
    if (overlayEl) {
      overlayEl.querySelector('.solvo-overlay-text').textContent = text;
      return;
    }

    overlayEl = document.createElement('div');
    overlayEl.className = 'solvo-overlay';
    overlayEl.innerHTML = `
      <div class="solvo-overlay-inner">
        <div class="solvo-overlay-spinner"></div>
        <span class="solvo-overlay-text">${text}</span>
      </div>
    `;
    document.body.appendChild(overlayEl);
  }

  function updateSolvoOverlay(text) {
    if (overlayEl) {
      overlayEl.querySelector('.solvo-overlay-text').textContent = text;
    }
  }

  function hideSolvoOverlay() {
    if (overlayEl) {
      overlayEl.remove();
      overlayEl = null;
    }
  }

  function showSolvoComplete(filled, total) {
    const el = document.createElement('div');
    el.className = 'solvo-toast';
    el.innerHTML = `
      <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="#0F7B6C" stroke-width="2.5">
        <path d="M20 6L9 17l-5-5"/>
      </svg>
      <span>Solvo filled ${filled}/${total} questions</span>
    `;
    document.body.appendChild(el);
    setTimeout(() => el.remove(), 5000);
  }

})();
