// =============================================
// SOLVO — Background Service Worker
// Handles tab management, AI queries, solve
// orchestration (persists when popup closes),
// and notifications.
// =============================================

// ---- Solve State (in-memory, survives popup close) ----
let solveState = {
  active: false,
  tabId: null,
  progress: { title: '', detail: '', current: 0, total: 0 },
  result: null,
  completed: false,
  pagesSolved: 0,
};
const MAX_PAGES = 20; // loop guard

// ---- Message Router ----
// Uses async wrapper so every code-path calls sendResponse exactly once,
// preventing "message port closed" warnings.
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  (async () => {
    try {
      switch (msg.action) {
        case 'fillForm':
          sendResponse(await handleFillForm(msg.tabId, msg.autoSubmit));
          return;

        case 'fillByLink':
          sendResponse(await handleFillByLink(msg.url, msg.autoSubmit));
          return;

        case 'startSolving':
          if (solveState.active) {
            sendResponse({ started: false, reason: 'Already solving' });
          } else {
            startSolvingInBackground(msg.tabId, msg.autoSubmit);
            sendResponse({ started: true });
          }
          return;

        case 'startSolvingByLink':
          if (solveState.active) {
            sendResponse({ started: false, reason: 'Already solving' });
          } else {
            startSolvingByLink(msg.url, msg.autoSubmit);
            sendResponse({ started: true });
          }
          return;

        case 'getSolveStatus':
          sendResponse({ ...solveState });
          return;

        case 'cancelSolving':
          solveState.active = false;
          solveState.completed = true;
          solveState.result = { success: false, error: 'Cancelled by user' };
          await chrome.storage.local.remove('solvingTabId');
          sendResponse({ ok: true });
          return;

        case 'clearSolveResult':
          solveState.completed = false;
          solveState.result = null;
          sendResponse({ ok: true });
          return;

        case 'queryGemini':
        case 'queryAI':
          sendResponse(await handleGeminiQuery(msg.questions, msg.profile));
          return;

        case 'getProfile': {
          const data = await chrome.storage.local.get(['profile']);
          sendResponse(data.profile || {});
          return;
        }

        case 'clearSolvingState':
          solveState.active = false;
          solveState.completed = false;
          solveState.result = null;
          solveState.pagesSolved = 0;
          await chrome.storage.local.remove('solvingTabId');
          sendResponse({ ok: true });
          return;

        case 'progressUpdate':
          solveState.progress = {
            title: msg.title || solveState.progress.title,
            detail: msg.detail || solveState.progress.detail,
            current: msg.current ?? solveState.progress.current,
            total: msg.total ?? solveState.progress.total,
          };
          // no response expected, but close the port cleanly
          sendResponse({ ok: true });
          return;

        case 'checkForUpdate':
          sendResponse(await checkForUpdate());
          return;

        case 'getUpdateInfo': {
          const updateData = await chrome.storage.local.get(['updateAvailable', 'updateVersion', 'updateDownloadUrl', 'updateChangelog', 'updateDismissedVersion']);
          sendResponse(updateData);
          return;
        }

        case 'dismissUpdate': {
          // User dismissed the update banner — remember which version they dismissed
          const dismissData = await chrome.storage.local.get(['updateVersion']);
          if (dismissData.updateVersion) {
            await chrome.storage.local.set({ updateDismissedVersion: dismissData.updateVersion });
          }
          sendResponse({ ok: true });
          return;
        }

        case 'openUpdatePage': {
          const upInfo = await chrome.storage.local.get(['updateVersion', 'updateDownloadUrl', 'updateChangelog']);
          const currentVersion = chrome.runtime.getManifest().version;
          const updatePageUrl = chrome.runtime.getURL('pages/update.html') +
            `?current=${encodeURIComponent(currentVersion)}` +
            `&latest=${encodeURIComponent(upInfo.updateVersion || '')}` +
            `&download=${encodeURIComponent(upInfo.updateDownloadUrl || 'https://solvo.pages.dev/download')}` +
            (upInfo.updateChangelog ? `&changelog=${encodeURIComponent(JSON.stringify(upInfo.updateChangelog))}` : '');
          chrome.tabs.create({ url: updatePageUrl });
          sendResponse({ ok: true });
          return;
        }

        case 'openVersionsPage': {
          chrome.tabs.create({ url: chrome.runtime.getURL('pages/versions.html') });
          sendResponse({ ok: true });
          return;
        }

        default:
          sendResponse({});
          return;
      }
    } catch (err) {
      console.error('[Solvo BG] Message handler error:', err);
      sendResponse({ error: err.message });
    }
  })();
  return true; // keep channel open for the async wrapper
});

// =============================================
// BACKGROUND SOLVE ORCHESTRATION
// Runs the full solve flow so it survives popup close
// =============================================

async function startSolvingInBackground(tabId, autoSubmit) {
  solveState = {
    active: true,
    tabId,
    progress: { title: 'Solving form...', detail: 'Extracting questions...', current: 0, total: 0 },
    result: null,
    completed: false,
    pagesSolved: 0,
  };

  await chrome.storage.local.set({ solvingTabId: tabId });

  try {
    await solveTabFormBg(tabId, autoSubmit);
  } catch (err) {
    console.error('[Solvo BG] Solve error:', err);
    solveState.result = { success: false, error: err.message };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Error', 'Failed to solve form: ' + err.message);
  }
}

async function startSolvingByLink(url, autoSubmit) {
  solveState = {
    active: true,
    tabId: null,
    progress: { title: 'Opening form...', detail: 'Working in background tab...', current: 0, total: 0 },
    result: null,
    completed: false,
    pagesSolved: 0,
  };

  try {
    const tab = await chrome.tabs.create({ url, active: false });
    solveState.tabId = tab.id;
    await chrome.storage.local.set({ solvingTabId: tab.id });

    await waitForTabLoad(tab.id);
    await sleep(2500);

    await solveTabFormBg(tab.id, autoSubmit);

    if (autoSubmit && solveState.result?.success) {
      await sleep(2000);
      try { await chrome.tabs.remove(tab.id); } catch {}
    }
  } catch (err) {
    console.error('[Solvo BG] Link solve error:', err);
    solveState.result = { success: false, error: err.message };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Error', 'Failed to solve form: ' + err.message);
  }
}

async function solveTabFormBg(tabId, autoSubmit) {
  if (!solveState.active) return; // cancelled

  // Loop guard
  if (solveState.pagesSolved >= MAX_PAGES) {
    console.warn('[Solvo BG] Max pages reached, stopping');
    solveState.result = { success: false, error: 'Max page limit reached — possible loop detected' };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Stopped', 'Reached max page limit. Possible loop detected.');
    return;
  }

  solveState.progress = { title: 'Solving form...', detail: 'Extracting questions...', current: 0, total: 0 };

  let extracted;
  try {
    extracted = await chrome.tabs.sendMessage(tabId, { action: 'extractQuestions' });
  } catch (err) {
    solveState.result = { success: false, error: 'Cannot communicate with form page. Make sure the Google Form tab is still open.' };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Error', 'Lost connection to form tab.');
    return;
  }

  if (!extracted?.success) {
    solveState.result = { success: false, error: extracted?.error || 'Failed to extract questions' };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Error', extracted?.error || 'No questions found.');
    return;
  }

  if (!solveState.active) return; // cancelled

  const { quizQuestions, personalAnswers, totalQuestions } = extracted;
  solveState.progress = { title: 'Solving form...', detail: `${quizQuestions.length} quiz questions found`, current: 0, total: quizQuestions.length };

  // Get AI answers
  let aiAnswers = [];
  if (quizQuestions.length > 0) {
    aiAnswers = await getAIAnswersBg(quizQuestions);
    if (!aiAnswers) {
      solveState.result = { success: false, error: 'All AI providers failed. Try again later.' };
      solveState.completed = true;
      solveState.active = false;
      await chrome.storage.local.remove('solvingTabId');
      sendNotification('Solvo — Error', 'AI providers failed.');
      return;
    }
  }

  if (!solveState.active) return; // cancelled

  // Merge answers
  const allAnswers = [];
  for (const [idx, ans] of Object.entries(personalAnswers)) {
    allAnswers.push({ ...ans, questionIndex: parseInt(idx) });
  }
  aiAnswers.forEach(a => {
    const quizQ = quizQuestions[a.questionIndex];
    if (quizQ) {
      allAnswers.push({ ...a, questionIndex: quizQ.index });
    }
  });

  // Fill
  solveState.progress = { title: 'Filling answers...', detail: 'Typing into form fields...', current: 0, total: totalQuestions };

  let fillResult;
  try {
    fillResult = await chrome.tabs.sendMessage(tabId, {
      action: 'fillWithAnswers',
      answers: allAnswers,
      autoSubmit,
    });
  } catch (err) {
    solveState.result = { success: false, error: 'Lost connection to form tab during fill.' };
    solveState.completed = true;
    solveState.active = false;
    await chrome.storage.local.remove('solvingTabId');
    sendNotification('Solvo — Error', 'Lost connection to form tab.');
    return;
  }

  // Handle multi-page forms (with loop guard)
  if (fillResult?.hasNextPage && solveState.active) {
    solveState.pagesSolved++;
    await sleep(2000);
    await solveTabFormBg(tabId, autoSubmit);
    return;
  }

  // Done!
  solveState.active = false;
  solveState.completed = true;
  await chrome.storage.local.remove('solvingTabId');

  if (fillResult?.success) {
    solveState.result = { success: true, filled: fillResult.filled, total: fillResult.total };
    sendNotification('Solvo — Complete!', `Filled ${fillResult.filled}/${fillResult.total} questions successfully.`);
  } else {
    solveState.result = { success: false, error: fillResult?.error || 'Failed to fill form' };
    sendNotification('Solvo — Error', fillResult?.error || 'Failed to fill form.');
  }
}

// =============================================
// AI ANSWER PIPELINE — Multi-provider (runs in BG)
// =============================================

const PROVIDER_INFO_BG = {
  free:       { label: 'Free AI',      needsKey: false },
  gemini:     { label: 'Gemini',       needsKey: true },
  groq:       { label: 'Groq',         needsKey: true },
  openrouter: { label: 'OpenRouter',   needsKey: true },
  together:   { label: 'Together AI',  needsKey: true },
};

async function getAIAnswersBg(quizQuestions) {
  const data = await chrome.storage.local.get([
    'provider', 'apiKey', 'model', 'batchSize', 'batchDelay',
    'geminiKey', 'groqKey', 'openrouterKey', 'togetherKey',
  ]);
  const provider = data.provider || 'free';
  const batchSize = parseInt(data.batchSize) || 5;
  const batchDelay = parseInt(data.batchDelay) || 1200;

  const batches = [];
  for (let i = 0; i < quizQuestions.length; i += batchSize) {
    batches.push(quizQuestions.slice(i, i + batchSize));
  }

  console.log(`[Solvo BG] ${quizQuestions.length} questions → ${batches.length} batch(es), provider=${provider}`);

  const allAnswers = [];
  let providerExhausted = false;

  for (let b = 0; b < batches.length; b++) {
    if (!solveState.active) return null; // cancelled

    const batch = batches[b];
    const batchLabel = `Batch ${b + 1}/${batches.length}`;
    solveState.progress = { title: 'Solving form...', detail: `${batchLabel} — ${batch.length} questions`, current: b, total: batches.length };

    let batchAnswers = null;

    if (provider !== 'free' && !providerExhausted) {
      solveState.progress.detail = `${batchLabel} — querying ${PROVIDER_INFO_BG[provider]?.label}...`;
      try {
        batchAnswers = await queryProviderBg(provider, batch, data);
      } catch (err) {
        console.warn(`[Solvo BG] ${provider} error:`, err.message);
        if (err.message.includes('429') || err.message.includes('rate')) {
          providerExhausted = true;
        }
      }
    }

    if (!batchAnswers) {
      solveState.progress.detail = `${batchLabel} — using free AI...`;
      batchAnswers = await queryFreeAIBg(batch);
      if (!batchAnswers) return null;
    }

    batchAnswers.forEach(a => {
      allAnswers.push({ ...a, questionIndex: a.questionIndex + (b * batchSize) });
    });

    if (b < batches.length - 1) {
      await sleep(batchDelay + Math.random() * 600);
    }
  }

  return allAnswers;
}

async function queryProviderBg(provider, questions, data) {
  switch (provider) {
    case 'gemini':   return await queryGeminiBg(questions, data);
    case 'groq':     return await queryGroqBg(questions, data);
    case 'openrouter': return await queryOpenRouterBg(questions, data);
    case 'together': return await queryTogetherBg(questions, data);
    default:         return await queryFreeAIBg(questions);
  }
}

function buildQuizPromptBg(questions) {
  let prompt = `Answer ALL of the following quiz/form questions with high accuracy.

RULES:
- Multiple choice: respond with EXACT text of the correct option.
- Checkboxes (select multiple): respond with each correct option.
- Short/long answers: concise, accurate answer.
- Dropdown: respond with EXACT option text.
- Linear scale: just the number.
- Never leave blank.

RESPOND IN THIS EXACT JSON FORMAT ONLY:
{"answers":[{"questionIndex":0,"answer":"your answer","type":"single|multiple|text"}]}
For checkboxes use array: "answer": ["Option 1", "Option 2"]

QUESTIONS:
`;

  questions.forEach((q, i) => {
    prompt += `\n--- Question ${i + 1} (Index ${i}) ---\n`;
    prompt += `Text: ${q.text}\n`;
    prompt += `Type: ${q.type}\n`;
    if (q.required) prompt += `Required: Yes\n`;
    if (q.options?.length > 0) {
      prompt += `Options:\n`;
      q.options.forEach((opt, j) => prompt += `  ${String.fromCharCode(65 + j)}) ${opt}\n`);
    }
    if (q.description) prompt += `Description: ${q.description}\n`;
  });

  prompt += `\nRespond ONLY with the JSON object, nothing else.`;
  return prompt;
}

function parseAIResponseBg(text) {
  try {
    let cleaned = text.trim();
    // Strip markdown code fences
    if (cleaned.startsWith('```')) {
      cleaned = cleaned.replace(/^```(?:json)?\s*\n?/, '').replace(/\n?```\s*$/, '');
    }
    // Strip <think>...</think> blocks (DeepSeek R1, etc.)
    cleaned = cleaned.replace(/<think>[\s\S]*?<\/think>/gi, '').trim();
    // Strip any leading/trailing non-JSON text
    const jsonMatch = cleaned.match(/\{[\s\S]*"answers"[\s\S]*\}/);
    if (jsonMatch) {
      const parsed = JSON.parse(jsonMatch[0]);
      return parsed.answers || [];
    }
    const parsed = JSON.parse(cleaned);
    return parsed.answers || [];
  } catch (err) {
    console.error('[Solvo BG] Failed to parse AI response:', err);
    return [];
  }
}

// ---- Gemini ----
const GEMINI_MODELS_BG = [
  'gemini-2.0-flash',
  'gemini-2.0-flash-lite',
  'gemini-1.5-flash',
  'gemini-1.5-flash-8b',
  'gemini-1.5-pro',
];

async function queryGeminiBg(questions, data) {
  const apiKey = data.geminiKey || data.apiKey;
  if (!apiKey) throw new Error('No Gemini API key configured');

  const preferredModel = data.model || 'gemini-2.0-flash';
  const modelsToTry = [preferredModel, ...GEMINI_MODELS_BG.filter(m => m !== preferredModel)];
  const prompt = buildQuizPromptBg(questions);

  for (const model of modelsToTry) {
    try {
      const resp = await fetch(
        `https://generativelanguage.googleapis.com/v1beta/models/${model}:generateContent?key=${apiKey}`,
        {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            contents: [{ parts: [{ text: prompt }] }],
            generationConfig: { temperature: 0.1, topP: 0.8, maxOutputTokens: 8192 },
          }),
        }
      );
      if (resp.status === 429) { await sleep(500); continue; }
      if (!resp.ok) continue;
      const result = await resp.json();
      const text = result.candidates?.[0]?.content?.parts?.[0]?.text;
      if (!text) continue;
      const answers = parseAIResponseBg(text);
      if (answers.length > 0) return answers;
    } catch (err) {
      console.warn(`[Solvo BG] Gemini ${model}:`, err.message);
    }
  }
  throw new Error('All Gemini models exhausted (429)');
}

// ---- Groq ----
const GROQ_MODELS_BG = [
  'llama-3.3-70b-versatile',
  'llama-3.1-8b-instant',
  'mixtral-8x7b-32768',
  'gemma2-9b-it',
];

async function queryGroqBg(questions, data) {
  const apiKey = data.groqKey;
  if (!apiKey) throw new Error('No Groq API key configured');

  const preferredModel = data.model || 'llama-3.3-70b-versatile';
  const modelsToTry = [preferredModel, ...GROQ_MODELS_BG.filter(m => m !== preferredModel)];
  const prompt = buildQuizPromptBg(questions);

  for (const model of modelsToTry) {
    try {
      const resp = await fetch('https://api.groq.com/openai/v1/chat/completions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${apiKey}` },
        body: JSON.stringify({
          model,
          messages: [
            { role: 'system', content: 'You are an expert quiz solver. Respond ONLY with valid JSON.' },
            { role: 'user', content: prompt },
          ],
          temperature: 0.1, max_tokens: 8192,
          response_format: { type: 'json_object' },
        }),
      });
      if (resp.status === 429) { await sleep(1000); continue; }
      if (!resp.ok) continue;
      const result = await resp.json();
      const text = result.choices?.[0]?.message?.content;
      if (!text) continue;
      const answers = parseAIResponseBg(text);
      if (answers.length > 0) return answers;
    } catch (err) {
      console.warn(`[Solvo BG] Groq ${model}:`, err.message);
    }
  }
  throw new Error('All Groq models exhausted');
}

// ---- OpenRouter ----
const OPENROUTER_MODELS_BG = [
  'meta-llama/llama-3.3-70b-instruct:free',
  'google/gemini-2.0-flash-exp:free',
  'deepseek/deepseek-r1:free',
  'qwen/qwen-2.5-72b-instruct:free',
];

async function queryOpenRouterBg(questions, data) {
  const apiKey = data.openrouterKey;
  if (!apiKey) throw new Error('No OpenRouter API key configured');

  const preferredModel = data.model || 'meta-llama/llama-3.3-70b-instruct:free';
  const modelsToTry = [preferredModel, ...OPENROUTER_MODELS_BG.filter(m => m !== preferredModel)];
  const prompt = buildQuizPromptBg(questions);

  for (const model of modelsToTry) {
    try {
      const resp = await fetch('https://openrouter.ai/api/v1/chat/completions', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${apiKey}`,
          'HTTP-Referer': chrome.runtime.getURL(''),
          'X-Title': 'Solvo Form Solver',
        },
        body: JSON.stringify({
          model,
          messages: [
            { role: 'system', content: 'You are an expert quiz solver. Respond ONLY with valid JSON. Do not wrap in markdown or code blocks. Do not include any thinking or reasoning text.' },
            { role: 'user', content: prompt },
          ],
          temperature: 0.1,
          max_tokens: 8192,
        }),
      });
      if (resp.status === 429) { await sleep(1000); continue; }
      if (resp.status === 401) throw new Error('Invalid OpenRouter API key');
      if (resp.status === 402) throw new Error('OpenRouter: insufficient credits');
      if (!resp.ok) {
        const errBody = await resp.text().catch(() => '');
        console.warn(`[Solvo BG] OpenRouter ${model} HTTP ${resp.status}:`, errBody);
        continue;
      }
      const result = await resp.json();
      const text = result.choices?.[0]?.message?.content;
      if (!text) continue;
      const answers = parseAIResponseBg(text);
      if (answers.length > 0) return answers;
    } catch (err) {
      console.warn(`[Solvo BG] OpenRouter ${model}:`, err.message);
    }
  }
  throw new Error('All OpenRouter models exhausted');
}

// ---- Together AI ----
const TOGETHER_MODELS_BG = [
  'meta-llama/Llama-3.3-70B-Instruct-Turbo',
  'Qwen/Qwen2.5-72B-Instruct-Turbo',
  'deepseek-ai/DeepSeek-R1-Distill-Llama-70B',
];

async function queryTogetherBg(questions, data) {
  const apiKey = data.togetherKey;
  if (!apiKey) throw new Error('No Together API key configured');

  const preferredModel = data.model || 'meta-llama/Llama-3.3-70B-Instruct-Turbo';
  const modelsToTry = [preferredModel, ...TOGETHER_MODELS_BG.filter(m => m !== preferredModel)];
  const prompt = buildQuizPromptBg(questions);

  for (const model of modelsToTry) {
    try {
      const resp = await fetch('https://api.together.xyz/v1/chat/completions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'Authorization': `Bearer ${apiKey}` },
        body: JSON.stringify({
          model,
          messages: [
            { role: 'system', content: 'You are an expert quiz solver. Respond ONLY with valid JSON.' },
            { role: 'user', content: prompt },
          ],
          temperature: 0.1, max_tokens: 8192,
        }),
      });
      if (resp.status === 429) { await sleep(1000); continue; }
      if (!resp.ok) continue;
      const result = await resp.json();
      const text = result.choices?.[0]?.message?.content;
      if (!text) continue;
      const answers = parseAIResponseBg(text);
      if (answers.length > 0) return answers;
    } catch (err) {
      console.warn(`[Solvo BG] Together ${model}:`, err.message);
    }
  }
  throw new Error('All Together models exhausted');
}

// ---- Free AI (Pollinations) ----
const FREE_AI_MAX_RETRIES_BG = 3;

async function queryFreeAIBg(questions) {
  const prompt = buildQuizPromptBg(questions);

  for (let attempt = 1; attempt <= FREE_AI_MAX_RETRIES_BG; attempt++) {
    try {
      const controller = new AbortController();
      const timeout = setTimeout(() => controller.abort(), 60000);

      const resp = await fetch('https://text.pollinations.ai/openai/chat/completions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        signal: controller.signal,
        body: JSON.stringify({
          messages: [
            { role: 'system', content: 'You are an expert quiz solver. Respond ONLY with valid JSON, no markdown.' },
            { role: 'user', content: prompt },
          ],
          model: 'openai',
          seed: Math.floor(Math.random() * 100000),
        }),
      });

      clearTimeout(timeout);

      if (resp.status === 429) { await sleep(2000 * attempt); continue; }
      if (!resp.ok) { await sleep(2000); continue; }

      const json = await resp.json();
      const content = json?.choices?.[0]?.message?.content;
      if (!content) continue;

      const answers = parseAIResponseBg(content);
      if (answers.length > 0) return answers;
    } catch (err) {
      console.warn(`[Solvo BG] Free AI attempt ${attempt}:`, err.message);
      if (attempt < FREE_AI_MAX_RETRIES_BG) await sleep(2000);
    }
  }
  return null;
}

// =============================================
// NOTIFICATIONS
// =============================================

function sendNotification(title, message) {
  try {
    chrome.notifications.create('solvo-' + Date.now(), {
      type: 'basic',
      iconUrl: 'icons/icon128.png',
      title,
      message,
      priority: 2,
    });
  } catch (err) {
    console.warn('[Solvo BG] Notification error:', err);
  }
}

// =============================================
// LEGACY HANDLERS (kept for backward compat)
// =============================================

// ---- Fill form on current tab ----
async function handleFillForm(tabId, autoSubmit) {
  try {
    const response = await chrome.tabs.sendMessage(tabId, {
      action: 'startFilling',
      autoSubmit,
    });
    return response;
  } catch (err) {
    return { success: false, error: err.message };
  }
}

// ---- Fill form by link (silent background tab) ----
async function handleFillByLink(url, autoSubmit) {
  try {
    const tab = await chrome.tabs.create({ url, active: false });
    await waitForTabLoad(tab.id);
    await sleep(2500);

    const response = await chrome.tabs.sendMessage(tab.id, {
      action: 'startFilling',
      autoSubmit,
    });

    if (autoSubmit && response?.success) {
      await sleep(2000);
      try { await chrome.tabs.remove(tab.id); } catch {}
    }

    return response;
  } catch (err) {
    return { success: false, error: err.message };
  }
}

// ---- Gemini API (legacy, still used by content.js startFilling) ----
const FALLBACK_MODELS = [
  'gemini-2.0-flash',
  'gemini-2.0-flash-lite',
  'gemini-1.5-flash',
  'gemini-1.5-flash-8b',
  'gemini-1.5-pro',
];

async function handleGeminiQuery(questions, profile) {
  const data = await chrome.storage.local.get(['apiKey', 'model', 'geminiKey']);
  const apiKey = data.geminiKey || data.apiKey;
  const preferredModel = data.model || 'gemini-2.0-flash';

  if (!apiKey) {
    return { success: false, error: 'No API key configured', needsFallback: true };
  }

  const modelsToTry = [preferredModel, ...FALLBACK_MODELS.filter(m => m !== preferredModel)];
  const prompt = buildPrompt(questions, profile);
  const bodyObj = {
    contents: [{ parts: [{ text: prompt }] }],
    generationConfig: {
      temperature: 0.1,
      topP: 0.8,
      maxOutputTokens: 8192,
    },
  };

  for (const model of modelsToTry) {
    try {
      const url = `https://generativelanguage.googleapis.com/v1beta/models/${model}:generateContent?key=${apiKey}`;
      const response = await fetch(url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(bodyObj),
      });

      if (response.status === 429) { await sleep(500); continue; }
      if (!response.ok) continue;

      const result = await response.json();
      const text = result.candidates?.[0]?.content?.parts?.[0]?.text;
      if (!text) continue;

      const answers = parseGeminiResponse(text);
      return { success: true, answers, model };
    } catch (err) {
      continue;
    }
  }

  return { success: false, needsFallback: true, error: 'All Gemini models rate limited.' };
}

function buildPrompt(questions, profile) {
  let prompt = `You are an expert at answering quiz and form questions with high accuracy. Answer ALL of the following questions.

RULES:
- For multiple choice: respond with the EXACT text of the correct option.
- For checkboxes (select multiple): respond with each correct option on a separate line.
- For short/long answers: give a concise, accurate answer.
- For dropdown questions: respond with the EXACT text of the correct option.
- For linear scale: respond with just the number.
- If you're not sure, give your best educated answer — never leave blank.

RESPOND IN THIS EXACT JSON FORMAT:
{
  "answers": [
    {
      "questionIndex": 0,
      "answer": "your answer here",
      "type": "single" | "multiple" | "text"
    }
  ]
}

For "multiple" type (checkboxes), use an array: "answer": ["Option 1", "Option 2"]

QUESTIONS:
`;

  questions.forEach((q, i) => {
    prompt += `\n--- Question ${i + 1} (Index ${i}) ---\n`;
    prompt += `Text: ${q.text}\n`;
    prompt += `Type: ${q.type}\n`;
    if (q.required) prompt += `Required: Yes\n`;
    if (q.options && q.options.length > 0) {
      prompt += `Options:\n`;
      q.options.forEach((opt, j) => {
        prompt += `  ${String.fromCharCode(65 + j)}) ${opt}\n`;
      });
    }
    if (q.description) {
      prompt += `Description: ${q.description}\n`;
    }
  });

  prompt += `\nRespond ONLY with the JSON object, nothing else.`;
  return prompt;
}

function parseGeminiResponse(text) {
  try {
    let cleaned = text.trim();
    if (cleaned.startsWith('```')) {
      cleaned = cleaned.replace(/^```(?:json)?\s*\n?/, '').replace(/\n?```\s*$/, '');
    }
    // Strip <think>...</think> blocks
    cleaned = cleaned.replace(/<think>[\s\S]*?<\/think>/gi, '').trim();
    const jsonMatch = cleaned.match(/\{[\s\S]*"answers"[\s\S]*\}/);
    if (jsonMatch) {
      const parsed = JSON.parse(jsonMatch[0]);
      return parsed.answers || [];
    }
    const parsed = JSON.parse(cleaned);
    return parsed.answers || [];
  } catch (err) {
    console.error('Failed to parse Gemini response:', err, text);
    return [];
  }
}

// ---- Helpers ----
function waitForTabLoad(tabId) {
  return new Promise((resolve) => {
    function listener(id, changeInfo) {
      if (id === tabId && changeInfo.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        resolve();
      }
    }
    chrome.tabs.onUpdated.addListener(listener);
    setTimeout(() => {
      chrome.tabs.onUpdated.removeListener(listener);
      resolve();
    }, 30000);
  });
}

function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

// ---- Install handler ----
chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === 'install') {
    console.log('Solvo installed successfully');
    chrome.tabs.create({ url: 'https://solvo.pages.dev/install' });
  } else if (details.reason === 'update') {
    const currentVersion = chrome.runtime.getManifest().version;
    const previousVersion = details.previousVersion;
    console.log(`[Solvo] Updated from v${previousVersion} to v${currentVersion}`);
    // Show the updated page with version info
    const updatedUrl = chrome.runtime.getURL('pages/updated.html') +
      `?from=${encodeURIComponent(previousVersion)}&to=${encodeURIComponent(currentVersion)}`;
    chrome.tabs.create({ url: updatedUrl });
  }
  // Check for updates on install/update
  checkForUpdate().catch(() => {});
});

// ---- Periodic update check (every 6 hours) ----
const UPDATE_CHECK_INTERVAL = 6 * 60 * 60 * 1000; // 6 hours
const UPDATE_CHECK_URL = 'https://solvo.pages.dev/version.json';

// Check on service worker startup
checkForUpdate().catch(() => {});

// Set up periodic alarm for update checks
try {
  chrome.alarms.create('solvo-update-check', { periodInMinutes: 360 }); // 6 hours
  chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === 'solvo-update-check') {
      checkForUpdate().catch(() => {});
    }
  });
} catch(e) {
  // alarms API may not be available
  console.warn('[Solvo] Alarms API not available, using setTimeout fallback');
  setInterval(() => checkForUpdate().catch(() => {}), UPDATE_CHECK_INTERVAL);
}

async function checkForUpdate() {
  try {
    const currentVersion = chrome.runtime.getManifest().version;
    console.log(`[Solvo] Checking for updates... current version: ${currentVersion}`);

    const response = await fetch(UPDATE_CHECK_URL, {
      cache: 'no-cache',
      headers: { 'Accept': 'application/json' },
    });

    if (!response.ok) {
      console.warn('[Solvo] Update check failed:', response.status);
      return { hasUpdate: false, error: `HTTP ${response.status}` };
    }

    const data = await response.json();
    // Expected format:
    // {
    //   "version": "1.5.0",
    //   "download": "https://solvo.pages.dev/solvo-latest.zip",
    //   "changelog": [
    //     { "tag": "new", "text": "Self-update system" },
    //     { "tag": "fix", "text": "Bug fix for X" }
    //   ],
    //   "required": false,
    //   "minVersion": "1.0.0",
    //   "versions": [ { "version": "1.4.0", "date": "...", "download": "...", ... }, ... ]
    // }

    const latestVersion = data.version;
    if (!latestVersion) {
      return { hasUpdate: false, error: 'No version in response' };
    }

    const hasUpdate = isNewerVersion(currentVersion, latestVersion);
    console.log(`[Solvo] Latest version: ${latestVersion}, hasUpdate: ${hasUpdate}`);

    // Find latest version entry from versions array for download URL
    const latestEntry = (data.versions || []).find(v => v.version === latestVersion);
    const downloadUrl = (latestEntry && latestEntry.download) || data.download || 'https://solvo.pages.dev/download';

    if (hasUpdate) {
      await chrome.storage.local.set({
        updateAvailable: true,
        updateVersion: latestVersion,
        updateDownloadUrl: downloadUrl,
        updateChangelog: (latestEntry && latestEntry.changelog) || data.changelog || [],
        updateLastCheck: Date.now(),
      });

      // Show badge on extension icon
      try {
        chrome.action.setBadgeText({ text: '!' });
        chrome.action.setBadgeBackgroundColor({ color: '#EB5757' });
      } catch(e) {}

      return {
        hasUpdate: true,
        currentVersion,
        latestVersion,
        downloadUrl: downloadUrl,
        changelog: (latestEntry && latestEntry.changelog) || data.changelog,
      };
    } else {
      // Clear any stale update info
      await chrome.storage.local.set({
        updateAvailable: false,
        updateLastCheck: Date.now(),
      });
      try {
        chrome.action.setBadgeText({ text: '' });
      } catch(e) {}

      return { hasUpdate: false, currentVersion, latestVersion };
    }
  } catch (err) {
    console.warn('[Solvo] Update check error:', err.message);
    return { hasUpdate: false, error: err.message };
  }
}

// Compare semver strings: returns true if latest > current
function isNewerVersion(current, latest) {
  const c = current.split('.').map(Number);
  const l = latest.split('.').map(Number);
  for (let i = 0; i < Math.max(c.length, l.length); i++) {
    const cv = c[i] || 0;
    const lv = l[i] || 0;
    if (lv > cv) return true;
    if (lv < cv) return false;
  }
  return false;
}

// ---- HTTP-Level User-Agent Spoof (declarativeNetRequest) ----
// This modifies the actual HTTP User-Agent header sent to servers,
// so server-side browser checks are also bypassed.
const CHROME_VERSION = '131';
const CHROME_FULL_VERSION = '131.0.6778.139';
const CHROME_UA_STRING = `Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/${CHROME_FULL_VERSION} Safari/537.36`;
const BROWSER_SPOOF_RULE_ID = 9999;
const BROWSER_SPOOF_RULE_ID_2 = 9998;

async function applyBrowserSpoofHeaders(enabled) {
  try {
    // Always remove existing rules first
    await chrome.declarativeNetRequest.updateDynamicRules({
      removeRuleIds: [BROWSER_SPOOF_RULE_ID, BROWSER_SPOOF_RULE_ID_2],
    });

    if (enabled) {
      // Add rules to replace User-Agent and all Client Hints headers
      await chrome.declarativeNetRequest.updateDynamicRules({
        addRules: [
        {
          id: BROWSER_SPOOF_RULE_ID,
          priority: 1,
          action: {
            type: 'modifyHeaders',
            requestHeaders: [
              {
                header: 'User-Agent',
                operation: 'set',
                value: CHROME_UA_STRING,
              },
              {
                header: 'sec-ch-ua',
                operation: 'set',
                value: `"Chromium";v="${CHROME_VERSION}", "Not_A Brand";v="24", "Google Chrome";v="${CHROME_VERSION}"`,
              },
              {
                header: 'sec-ch-ua-mobile',
                operation: 'set',
                value: '?0',
              },
              {
                header: 'sec-ch-ua-platform',
                operation: 'set',
                value: '"Windows"',
              },
              {
                header: 'sec-ch-ua-full-version',
                operation: 'set',
                value: `"${CHROME_FULL_VERSION}"`,
              },
              {
                header: 'sec-ch-ua-full-version-list',
                operation: 'set',
                value: `"Chromium";v="${CHROME_FULL_VERSION}", "Not_A Brand";v="24.0.0.0", "Google Chrome";v="${CHROME_FULL_VERSION}"`,
              },
              {
                header: 'sec-ch-ua-arch',
                operation: 'set',
                value: '"x86"',
              },
              {
                header: 'sec-ch-ua-bitness',
                operation: 'set',
                value: '"64"',
              },
              {
                header: 'sec-ch-ua-model',
                operation: 'set',
                value: '""',
              },
              {
                header: 'sec-ch-ua-platform-version',
                operation: 'set',
                value: '"15.0.0"',
              },
              {
                header: 'sec-ch-ua-wow64',
                operation: 'set',
                value: '?0',
              },
            ],
          },
          condition: {
            urlFilter: '*',
            resourceTypes: [
              'main_frame', 'sub_frame', 'xmlhttprequest', 'script',
              'stylesheet', 'image', 'font', 'media', 'other',
            ],
          },
        },
        // Second rule to remove browser-specific headers that reveal Edge/Brave/Opera
        {
          id: BROWSER_SPOOF_RULE_ID_2,
          priority: 1,
          action: {
            type: 'modifyHeaders',
            requestHeaders: [
              { header: 'sec-ch-ua-edge', operation: 'remove' },
              { header: 'x-edge-shopping-flag', operation: 'remove' },
            ],
          },
          condition: {
            urlFilter: '*',
            resourceTypes: [
              'main_frame', 'sub_frame', 'xmlhttprequest', 'script',
              'stylesheet', 'image', 'font', 'media', 'other',
            ],
          },
        }],
      });
      console.log(`[Solvo] HTTP User-Agent spoof ENABLED — all requests now appear as Chrome ${CHROME_VERSION}`);
    } else {
      console.log('[Solvo] HTTP User-Agent spoof DISABLED');
    }
  } catch (err) {
    console.warn('[Solvo] Failed to apply browser spoof headers:', err.message);
  }
}

// Apply on startup based on stored setting
(async () => {
  const data = await chrome.storage.local.get(['enableBrowserSpoof']);
  if (data.enableBrowserSpoof) {
    applyBrowserSpoofHeaders(true);
    await registerSpoofContentScript();
  } else {
    await unregisterSpoofContentScript();
  }
})();

// Listen for setting changes and apply/remove header rules immediately
chrome.storage.onChanged.addListener(async (changes, area) => {
  if (area === 'local' && changes.enableBrowserSpoof) {
    const enabled = !!changes.enableBrowserSpoof.newValue;
    applyBrowserSpoofHeaders(enabled);
    if (enabled) {
      await registerSpoofContentScript();
    } else {
      await unregisterSpoofContentScript();
    }
  }
});

// ---- Dynamic Content Script Registration for Browser Spoof ----
// Using registerContentScripts ensures the spoof runs at TRUE document_start
// in the MAIN world — before any inline page scripts can read navigator.userAgent.
// This eliminates the race condition that programmatic executeScript had.
const SPOOF_SCRIPT_ID = 'solvo-browser-spoof';

async function registerSpoofContentScript() {
  try {
    // Remove first in case it already exists
    try { await chrome.scripting.unregisterContentScripts({ ids: [SPOOF_SCRIPT_ID] }); } catch (e) {}

    await chrome.scripting.registerContentScripts([{
      id: SPOOF_SCRIPT_ID,
      matches: ['<all_urls>'],
      js: ['content/spoof.js'],
      runAt: 'document_start',
      world: 'MAIN',
      allFrames: true,
      persistAcrossSessions: true,
    }]);
    console.log('[Solvo] Browser spoof content script REGISTERED (document_start, MAIN world, all frames)');
  } catch (err) {
    console.warn('[Solvo] Failed to register spoof content script:', err.message);
  }
}

async function unregisterSpoofContentScript() {
  try {
    await chrome.scripting.unregisterContentScripts({ ids: [SPOOF_SCRIPT_ID] });
    console.log('[Solvo] Browser spoof content script UNREGISTERED');
  } catch (err) {
    // Not registered — ignore
  }
}

// ---- Apply bypass features on tab navigation ----
// We use two injection phases:
//   1. EARLY (document_start, MAIN world): Tab switch + fullscreen bypass
//      Must run before ANY page scripts to intercept all event listeners.
//   2. LATE (document_idle): Right-click + copy/paste DOM cleanup
//      Needs the DOM to be ready to remove attributes and inject styles.

// Helper: check if URL is restricted (can't inject into)
function isRestrictedUrl(url) {
  if (!url) return true;
  return url.startsWith('chrome://') || url.startsWith('edge://') ||
         url.startsWith('chrome-extension://') || url.startsWith('extension://') ||
         url.startsWith('about:') || url.startsWith('chrome-search://') ||
         url.startsWith('devtools://') || url.startsWith('brave://');
}

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  // Skip restricted URLs
  if (isRestrictedUrl(tab.url)) return;

  try {
    const data = await chrome.storage.local.get(['enableRightClick', 'enableCopyPaste', 'enableTabSwitch', 'enableFullscreenBypass']);
    const needsEarly = data.enableTabSwitch || data.enableFullscreenBypass || data.enableCopyPaste;
    const needsLate = data.enableRightClick || data.enableCopyPaste;

    // Phase 1: Inject EARLY into MAIN world at document_start for API overrides
    // (Browser spoof is handled separately via registerContentScripts for guaranteed early execution)
    if (changeInfo.status === 'loading' && needsEarly) {
      await chrome.scripting.executeScript({
        target: { tabId, allFrames: true },
        world: 'MAIN',
        injectImmediately: true,
        func: applyEarlyBypass,
        args: [!!data.enableTabSwitch, !!data.enableFullscreenBypass, !!data.enableCopyPaste],
      });
    }

    // Phase 2: Inject LATE for DOM modifications (right-click, copy/paste)
    if (changeInfo.status === 'complete' && needsLate) {
      await chrome.scripting.executeScript({
        target: { tabId, allFrames: true },
        func: applyLateBypass,
        args: [!!data.enableRightClick, !!data.enableCopyPaste],
      });
    }
  } catch (err) {
    // Ignore errors for restricted pages
  }
});

// ---- Backup: Also inject on webNavigation.committed for sub_frame navigations ----
// Some assessment sites create iframes AFTER the main page loads, so
// tabs.onUpdated may miss them. webNavigation catches sub_frame navigations.
if (chrome.webNavigation) {
  chrome.webNavigation.onCommitted.addListener(async (details) => {
    // Only care about sub_frame (iframes); main_frame is handled by tabs.onUpdated
    if (details.frameId === 0) return;
    if (isRestrictedUrl(details.url)) return;

    try {
      const data = await chrome.storage.local.get(['enableTabSwitch', 'enableFullscreenBypass', 'enableCopyPaste']);
      const needsEarly = data.enableTabSwitch || data.enableFullscreenBypass || data.enableCopyPaste;
      if (!needsEarly) return;

      await chrome.scripting.executeScript({
        target: { tabId: details.tabId, frameIds: [details.frameId] },
        world: 'MAIN',
        injectImmediately: true,
        func: applyEarlyBypass,
        args: [!!data.enableTabSwitch, !!data.enableFullscreenBypass, !!data.enableCopyPaste],
      });
    } catch (err) {
      // Ignore errors for restricted frames
    }
  });
}

// ---- EARLY BYPASS: Runs in MAIN world before page scripts ----
// NOTE: Browser spoof is handled by a registered content script (content/spoof.js)
// for guaranteed document_start execution. This function handles other bypasses.
function applyEarlyBypass(tabSwitch, fullscreenBypass, copyPasteBypass) {
  if (window.__solvoEarlyBypassApplied) return;
  window.__solvoEarlyBypassApplied = true;

  // Collect ALL event types to block across features, then do a single addEventListener override
  const BLOCKED_WIN_DOC_EVENTS = new Set(); // events blocked on window/document
  const BLOCKED_ALL_EVENTS = new Set();     // events blocked on any target
  let interceptCopyPasteKeys = false;

  if (tabSwitch) {
    // === 1. Override Page Visibility API properties ===
    Object.defineProperty(document, 'hidden', {
      get: () => false,
      configurable: true,
    });
    Object.defineProperty(document, 'visibilityState', {
      get: () => 'visible',
      configurable: true,
    });
    try {
      Object.defineProperty(document, 'webkitHidden', { get: () => false, configurable: true });
      Object.defineProperty(document, 'webkitVisibilityState', { get: () => 'visible', configurable: true });
    } catch(e) {}

    // === 2. Override document.hasFocus() to always return true ===
    document.hasFocus = () => true;

    // === 3. Override on-property setters ===
    let _fakeOnVisChange = null;
    Object.defineProperty(document, 'onvisibilitychange', {
      get: () => _fakeOnVisChange,
      set: (fn) => { _fakeOnVisChange = fn; },
      configurable: true,
    });

    for (const target of [window, document]) {
      for (const prop of ['onblur', 'onfocus']) {
        let _fake = null;
        Object.defineProperty(target, prop, {
          get: () => _fake,
          set: (fn) => { _fake = fn; },
          configurable: true,
        });
      }
    }

    // Register events to block on window/document
    ['visibilitychange', 'webkitvisibilitychange', 'blur', 'focus',
     'focusin', 'focusout', 'pageshow', 'pagehide'].forEach(e => BLOCKED_WIN_DOC_EVENTS.add(e));

    // Capture-phase hard-block
    const hardBlock = (e) => { e.stopImmediatePropagation(); e.preventDefault(); };
    for (const evt of BLOCKED_WIN_DOC_EVENTS) {
      document.addEventListener(evt, hardBlock, true);
      window.addEventListener(evt, hardBlock, true);
    }
  }

  if (fullscreenBypass) {
    // === Fullscreen bypass: fake that we are always in fullscreen ===
    const fakeFullscreenEl = document.documentElement;

    // Override detection properties
    const fsProps = {
      'fullscreenElement': fakeFullscreenEl,
      'webkitFullscreenElement': fakeFullscreenEl,
      'mozFullScreenElement': fakeFullscreenEl,
      'msFullscreenElement': fakeFullscreenEl,
      'fullscreen': true,
      'webkitIsFullScreen': true,
      'mozFullScreen': true,
    };
    for (const [prop, val] of Object.entries(fsProps)) {
      try {
        Object.defineProperty(document, prop, { get: () => val, configurable: true });
      } catch(e) {}
    }

    // Make requestFullscreen a no-op
    const fakeReq = function() { return Promise.resolve(); };
    for (const m of ['requestFullscreen', 'webkitRequestFullscreen', 'webkitRequestFullScreen', 'mozRequestFullScreen', 'msRequestFullscreen']) {
      try { Element.prototype[m] = fakeReq; } catch(e) {}
    }

    // Make exitFullscreen a no-op
    for (const m of ['exitFullscreen', 'webkitExitFullscreen', 'mozCancelFullScreen', 'msExitFullscreen']) {
      try { document[m] = () => Promise.resolve(); } catch(e) {}
    }

    // Override on-property setters for fullscreen events
    for (const prop of ['onfullscreenchange', 'onwebkitfullscreenchange', 'onfullscreenerror', 'onwebkitfullscreenerror']) {
      let _fake = null;
      try {
        Object.defineProperty(document, prop, {
          get: () => _fake,
          set: (fn) => { _fake = fn; },
          configurable: true,
        });
      } catch(e) {}
    }

    // Fake screen dimensions
    try {
      Object.defineProperty(screen, 'availWidth', { get: () => window.innerWidth, configurable: true });
      Object.defineProperty(screen, 'availHeight', { get: () => window.innerHeight, configurable: true });
    } catch(e) {}

    // Register fullscreen events to block on ALL targets
    const fsEvents = [
      'fullscreenchange', 'webkitfullscreenchange', 'mozfullscreenchange', 'msfullscreenchange',
      'fullscreenerror', 'webkitfullscreenerror', 'mozfullscreenerror', 'msfullscreenerror',
    ];
    fsEvents.forEach(e => BLOCKED_ALL_EVENTS.add(e));

    // Capture-phase hard-block
    const fsBlock = (e) => { e.stopImmediatePropagation(); e.preventDefault(); };
    for (const evt of BLOCKED_ALL_EVENTS) {
      document.addEventListener(evt, fsBlock, true);
      window.addEventListener(evt, fsBlock, true);
    }
  }

  // === Copy/Paste keyboard shortcut bypass (Ctrl+C/V/X/A) ===
  if (copyPasteBypass) {
    interceptCopyPasteKeys = true;

    // Hard-block page keydown/keyup listeners from seeing Ctrl+C/V/X/A
    const COPY_PASTE_KEYS = new Set(['c', 'v', 'x', 'a']);
    const guardKeyEvent = (e) => {
      if ((e.ctrlKey || e.metaKey) && COPY_PASTE_KEYS.has(e.key?.toLowerCase())) {
        e.stopImmediatePropagation();
        // Do NOT call preventDefault — let the browser handle the native clipboard action
      }
    };
    document.addEventListener('keydown', guardKeyEvent, true);
    document.addEventListener('keyup', guardKeyEvent, true);
    window.addEventListener('keydown', guardKeyEvent, true);
    window.addEventListener('keyup', guardKeyEvent, true);

    // Also block beforeinput/input interception on clipboard operations
    ['copy', 'cut', 'paste'].forEach(evt => {
      document.addEventListener(evt, (e) => e.stopImmediatePropagation(), true);
      window.addEventListener(evt, (e) => e.stopImmediatePropagation(), true);
    });
  }

  // === Single unified addEventListener / removeEventListener override ===
  if (BLOCKED_WIN_DOC_EVENTS.size > 0 || BLOCKED_ALL_EVENTS.size > 0 || interceptCopyPasteKeys) {
    const origAdd = EventTarget.prototype.addEventListener;
    const origRemove = EventTarget.prototype.removeEventListener;

    // For copy/paste bypass: wrap keydown/keyup listeners so they can't see Ctrl+C/V/X/A
    const COPY_PASTE_KEYS = new Set(['c', 'v', 'x', 'a']);

    EventTarget.prototype.addEventListener = function(type, listener, options) {
      const t = (type || '').toLowerCase();
      if (BLOCKED_ALL_EVENTS.has(t)) return;
      if (BLOCKED_WIN_DOC_EVENTS.has(t) && (this === window || this === document)) return;

      // Wrap keydown/keyup listeners to skip Ctrl+C/V/X/A if copy/paste bypass is on
      if (interceptCopyPasteKeys && (t === 'keydown' || t === 'keyup') && typeof listener === 'function') {
        const origListener = listener;
        const wrappedListener = function(e) {
          if ((e.ctrlKey || e.metaKey) && COPY_PASTE_KEYS.has(e.key?.toLowerCase())) return;
          return origListener.call(this, e);
        };
        wrappedListener.__solvoOriginal = origListener;
        return origAdd.call(this, type, wrappedListener, options);
      }

      return origAdd.call(this, type, listener, options);
    };

    EventTarget.prototype.removeEventListener = function(type, listener, options) {
      const t = (type || '').toLowerCase();
      if (BLOCKED_ALL_EVENTS.has(t)) return;
      if (BLOCKED_WIN_DOC_EVENTS.has(t) && (this === window || this === document)) return;
      return origRemove.call(this, type, listener, options);
    };
  }
}

// ---- LATE BYPASS: Runs after DOM is ready for attribute/style cleanup ----
function applyLateBypass(rightClick, copyPaste) {
  if (window.__solvoLateBypassApplied) return;
  window.__solvoLateBypassApplied = true;

  if (rightClick) {
    // Enable right-click
    document.addEventListener('contextmenu', (e) => e.stopImmediatePropagation(), true);
    // Remove inline oncontextmenu handlers
    const removeContextMenu = () => {
      document.querySelectorAll('[oncontextmenu]').forEach(el => {
        el.removeAttribute('oncontextmenu');
      });
      if (document.body) document.body.oncontextmenu = null;
      if (document.documentElement) document.documentElement.oncontextmenu = null;
    };
    removeContextMenu();
    setTimeout(removeContextMenu, 2000);
    setTimeout(removeContextMenu, 5000);
  }

  if (copyPaste) {
    // Enable copy, cut, paste, select
    ['copy', 'cut', 'paste', 'selectstart', 'select'].forEach(evt => {
      document.addEventListener(evt, (e) => e.stopImmediatePropagation(), true);
    });
    const removeRestrictions = () => {
      document.querySelectorAll('[oncopy], [onpaste], [oncut], [onselectstart]').forEach(el => {
        el.removeAttribute('oncopy');
        el.removeAttribute('onpaste');
        el.removeAttribute('oncut');
        el.removeAttribute('onselectstart');
      });
      if (!document.querySelector('#solvo-bypass-style')) {
        const style = document.createElement('style');
        style.id = 'solvo-bypass-style';
        style.textContent = `
          * {
            -webkit-user-select: text !important;
            -moz-user-select: text !important;
            -ms-user-select: text !important;
            user-select: text !important;
          }
        `;
        (document.head || document.documentElement).appendChild(style);
      }
    };
    removeRestrictions();
    setTimeout(removeRestrictions, 2000);
  }
}

// ---- Clean up solving state on tab close ----
chrome.tabs.onRemoved.addListener(async (tabId) => {
  const data = await chrome.storage.local.get(['solvingTabId']);
  if (data.solvingTabId === tabId) {
    await chrome.storage.local.remove('solvingTabId');
    console.log('[Solvo] Solving tab closed, cleared state');
  }
});